/* Test suite for the Jacobi solver for Laplace equation.
 * Description: This file contains a series of unit test for the components
 *              of solver of the Laplace equation using Jacobi iterative method
 *              on a square grid, given the boundary conditions
 *
 * Author: G.P. Brandino
 * Copyright: 2019 eXact-lab s.r.l.
 * License: GPLv2
 */

#include "jacobi_functions.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

double test_update_small(){

    int i;
    size_t dimension = 1;
    size_t byte_dimension = sizeof(double*) * ( dimension + 2 ) * ( dimension + 2 );
    double* grid = (double*) malloc(byte_dimension);
    double* gridNew = (double*)malloc(byte_dimension);
        
    // We fill the 3x3 grid with 1 only in the elements that will be used by the update routine 
    // |  *  |
    // |*   *|
    // |  *  |
    // First we set everything to 0
    for (i=0; i < (dimension + 2)*(dimension + 2); i++) {
        grid[i] = 0.0;
    }
    // now we fill the 4 relevant elements
    grid[1] = 1.0;
    grid[3] = 1.0;
    grid[5] = 1.0;
    grid[7] = 1.0;

    // here the call of the routine we want to test
    update( grid, gridNew, dimension);
    
    // since the matrix is 3x3, the only element calculated is the central one
    return gridNew[4];

    free(grid);
    free(gridNew);
}


bool test_boundary_conditions(){

    int i,j;
    size_t dimension = 10;
    size_t byte_dimension = sizeof(double*) * ( dimension + 2 ) * ( dimension + 2 );
    double* grid = (double*) malloc(byte_dimension);
    double* gridNew = (double*)malloc(byte_dimension);
    bool equal = true;
    for (i = 0; i < ( dimension + 2) * ( dimension + 2 ); i++) { 
        grid[i] = 0.0;
        gridNew[i] = 0.0;
        }


    setBoundaryConditions( grid, gridNew, dimension);

    for (i = 0; i < dimension + 2; i++) {
        for (j = 0; j < dimension + 2; j++) {
            if (abs(grid[i * (dimension + 2) + j] - gridNew[i * (dimension + 2) + j] ) > 10e-15){
                return false;    
            }
        }
    }

    return true;

    free(grid);
    free(gridNew);

}


int main(int argc, char** argv){

    printf(" -- Starting tests \n\n");


    double testVal1 = test_update_small();

    if ( abs(testVal1 - 1) < 10e-15 ){
        printf( " Test update small: passed \n");
    } else {
        printf( " Test update small: failed! \n");
    }
    
    printf("\n");

    bool testVal2 = test_boundary_conditions();
    if ( testVal2 ){
        printf( " Test boundary condition: passed \n");
    } else {
        printf( " Test boundary condition: failed! \n");
    }

    return 0;

}
