##\file jacobi_functions.py
# \brief This file is a part of the Jacobi solver for Laplace equation.
# \details This file provides several routines for the solver of the
#              Laplace equation using Jacobi iterative method on a square grid,
#              given the boundary conditions
#
# \author: Giuseppe Piero Brandino - eXact-lab s.r.l.
# \copyright: 2019 eXact-lab s.r.l. GPLv2
#


def setBoundaryConditions(grid, gridNew, dimension):
    """ 
    Sets the boundary conditions on both the grid buffers
    Args: 
        grid double precision array containing the first grid buffer
        gridNew double precision array containing the second grid buffer
        dimension the size of the system. The buffers should be of size 
        (dimension + 2)^2
    """

    increment = 100.0 / (dimension + 1)
    for i in range(1,dimension + 2):
        grid[i][0] = i * increment
        grid[dimension + 1][ dimension + 1 - i]= i * increment
        gridNew[i][0] = i * increment
        gridNew[dimension + 1][dimension + 1 - i]= i * increment

    #for i in range(dimension + 2):
    #    grid[i][0]  = 100.0;
    #    grid[i][-1] = 100;
    #    grid[0][i]  = 100;
    #    grid[-1][i] = 100;
 
    #    gridNew[i][0]  = 100.0;
    #    gridNew[i][-1] = 100;
    #    gridNew[0][i]  = 100;
    #    gridNew[-1][i] = 100;

def update(grid, gridNew, dimension):
    """ Performs one step of the Jacobi iteravite process
    Args:
        grid double precision array containing the first grid buffer
        gridNew double precision array containing the second grid buffer
        dimension the size of the system. The buffers should be of size 
        (dimension + 2)^2
    """

    for i in range(1,dimension + 1):
        for j in range(1,dimension + 1):
            gridNew[i][j] = 0.25 * (grid[i-1][j] + grid[i+1][j] + grid[i][j-1] + grid[i][j+1])


def printOutput(grid, dimension):
    """ Prints the output in text format
    Args:
        grid double precision array containing the first grid buffer
        dimension the size of the system. The buffers should be of size 
        (dimension + 2)^2
    """

    outputFile = open("solution.dat","w+")

    for i in range(dimension+2):
        for j in range(dimension+2):
            outputFile.write(str(grid[i][j]) + " ")
        outputFile.write("\n")

    outputFile.close()

