/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-05-06
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTFORWARD_H
#define ABSTRACTFORWARD_H

#include <QGpCoreTools.h>
#include "DinverCoreDLLExport.h"
#include "RealSpace.h"
#include "ForwardSignal.h"
#include "SimpleCondition.h"

namespace DinverCore {

class UniqueRandom;
class ReportWriter;
class ScaledModels;
class VoronoiNavigator;
class ModelSet;

class DINVERCORE_EXPORT AbstractForward : public XMLClass
{
  TRANSLATIONS("AbstractForward")
public:
  AbstractForward();
  virtual ~AbstractForward();

  virtual AbstractForward * clone() const = 0;

  const RealSpace& parameterSpace() const { return _parameterSpace; }
  RealSpace& parameterSpace() { return _parameterSpace; }

  virtual void misfit();
  virtual double misfit( bool& ok );
  virtual double misfit( double * /*model*/ , bool& /*ok*/ ) { return 1.0; }
  virtual void writeReport( ReportWriter * /*report*/ ) {}
  virtual void valueChanged( const Parameter * p = 0 ) { Q_UNUSED(p); }
  virtual bool isFussyOk( const Parameter * /*p*/ ) { return true; }

  virtual void redirectStream() {}
  void setFinishSignal( ForwardSignal * s ) { _finishSignal = s; }
  void setGenerator( UniqueRandom * generator ) { _generator = generator; }
  void setAllModels( ModelSet * allModels ) { _allModels = allModels; }
  bool firstModel();
  void copyValues( const AbstractForward& o );
  void initGenerate( const ScaledModels * scaledModels, int nWalks );
  virtual bool generate( int parentActiveIndex = -1 );

  enum State { Sleeping, Repairing, Accepting, Running, Aborting, Sending };

  virtual int maximumThreadCount() const { return 1; }
  virtual bool wake();
  inline bool beginRepair();
  inline void endRepair();
  inline bool lock();
  inline void abort();
  inline void unlock();
  virtual void sleep();

  inline bool isSleeping() const;
  inline bool isAccepting() const;
  inline bool isRunning() const;
  inline bool isAborting() const;
  inline bool isSending() const;

  int parentActiveIndex() const { return _parentActiveIndex; }
  int modelIndex() const { return _modelIndex; }
  double misfitValue() const { return _misfitValue; }
  bool isValidMisfit() const { return _isValidMisfit; }
  bool isValidParent() const { return _isValidParent; }
  const QVector<int>& navigatorHits() const { return _navigatorHits; }

  void timeReport( const QString& prefix );
  // Make this interface public
  virtual void xml_writeProperties( XML_WRITEPROPERTIES_ARGS ) const { XMLClass::xml_writeProperties(s, context); }
  virtual void xml_writeChildren( XML_WRITECHILDREN_ARGS ) const { XMLClass::xml_writeChildren(s, context); }
  virtual XMLMember xml_member(XML_MEMBER_ARGS) { return XMLClass::xml_member(tag, attributes, context); }
  virtual bool xml_setProperty( XML_SETPROPERTY_ARGS ) { return XMLClass::xml_setProperty( memberID, attributes, content, context ); }
  virtual void xml_polish( XML_POLISH_ARGS ) { XMLClass::xml_polish(context); }
protected:
  bool generateInternal();
  void setParentActiveIndex( int parentActiveIndex ) { _parentActiveIndex = parentActiveIndex; }
  QString streamPrefix() const {return _streamPrefix.arg(_modelIndex);}
  void addTimeMisfit( int ms ) { _timeMisfit+=ms; }
  void addTimeWait( int ms ) { _timeWait+=ms; }
  inline void send( double misfit, bool isValid, bool isValidParent = true );
private:
  bool monteCarlo();
  bool neighborhood();

  RealSpace _parameterSpace;

  ModelSet * _allModels;
  UniqueRandom * _generator;
  VoronoiNavigator * _nav;
  int _nWalks;
  int _parentActiveIndex;
  int _modelIndex;
  QVector<int> _navigatorHits;

  mutable QMutex _stateMutex;
  State _state;
  ForwardSignal * _finishSignal;

  double _misfitValue;
  bool _isValidMisfit;
  bool _isValidParent;

  int _timeWait;
  int _timeGenerator;
  int _timeMisfit;
  static const QString _streamPrefix;
};

inline bool AbstractForward::lock()
{
  QMutexLocker ml (&_stateMutex);
  if ( _state == Accepting ) {
    _state = Running;
    return true;
  } else {
    return false;
  }
}

void AbstractForward::abort()
{
  _stateMutex.lock();
  switch (_state) {
  case Sleeping:
  case Accepting:
  case Repairing:
  case Aborting:
    _stateMutex.unlock();
    break;
  case Running:
    _state = Aborting;
    _stateMutex.unlock();
    break;
  case Sending:
    _state = Accepting;
    _stateMutex.unlock();
    _finishSignal->finished( this );
    break;
  }
}

void AbstractForward::unlock()
{
  QMutexLocker ml (&_stateMutex);
  switch (_state) {
  case Sleeping:
  case Repairing:
  case Accepting:
    break;
  case Aborting:
  case Running:
  case Sending:
    _state = Accepting;
    break;
  }
}

inline void AbstractForward::send( double misfit, bool isValid, bool isValidParent )
{
  _misfitValue = misfit;
  _isValidMisfit = isValid;
  _isValidParent = isValidParent;
  _stateMutex.lock();
  switch (_state) {
  case Sleeping:
  case Repairing:
  case Accepting:
  case Sending:
    ASSERT(false);
    break;
  case Aborting:
    _state = Accepting;
    _stateMutex.unlock();
    _finishSignal->finished( this );
    break;
  case Running:
    _state = Sending;
    _stateMutex.unlock();
    _finishSignal->finished( this );
    break;
  }
}

inline bool AbstractForward::beginRepair()
{
  QMutexLocker ml (&_stateMutex);
  if ( _state == Sleeping ) {
    _state = Repairing;
    return true;
  } else {
    return false;
  }
}

inline void AbstractForward::endRepair()
{
  QMutexLocker ml (&_stateMutex);
  switch (_state) {
  case Sleeping:
  case Repairing:
    _state = Sleeping;
    break;
  case Accepting:
  case Aborting:
  case Running:
  case Sending:
    break;
  }
}

inline bool AbstractForward::isSleeping() const
{
  QMutexLocker ml (&_stateMutex);
  return _state == Sleeping;
}

inline bool AbstractForward::isAccepting() const
{
  QMutexLocker ml (&_stateMutex);
  return _state == Accepting;
}

inline bool AbstractForward::isRunning() const
{
  QMutexLocker ml (&_stateMutex);
  return _state ==  Running;
}

inline bool AbstractForward::isAborting() const
{
  QMutexLocker ml (&_stateMutex);
  return _state == Aborting;
}

inline bool AbstractForward::isSending() const
{
  QMutexLocker ml (&_stateMutex);
  return _state == Sending;
}

} // namespace DinverCore

#endif // ABSTRACTFORWARD_H
