/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-05-13
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MODELSET_H
#define MODELSET_H

#include <QGpCoreTools.h>

namespace DinverCore {

class Model;
class RealSpace;

class ModelSet : private QSet<Model>, private IncreaseStorage
{
  TRANSLATIONS("ModelSet")
public:
  ModelSet(int parameterCount, int targetCount, int defaultCapacity = 1024 );
  ~ModelSet();

  bool importModels(RealSpace& parameterSpace, QString fileName, bool strict=true);
  int add(const RealSpace& parameterSpace, const double * misfit=0);

  void setMisfit(int modelIndex, int targetIndex, double m) {_misfits[_targetCount*modelIndex+targetIndex]=m;}
  void clear();

  void reserve( int n ) {IncreaseStorage::reserve(n);}
  int count() const {return IncreaseStorage::size();}
  int parameterCount() const {return _parameterCount;}
  int targetCount() const {return _targetCount;}

  const int * model(int modelIndex) const {return _parameters+_parameterCount*modelIndex;}
  const int * firstModel() const {return _parameters;}
  const double * misfit(int modelIndex) const {return _misfits+_targetCount*modelIndex;}

  void lock() {_modelMutex.lock();}
  void unlock() {_modelMutex.unlock();}

  void print(int modelIndex) const;
  int bestModel() const;
private:
  bool importModelsFromAscii(RealSpace& parameterSpace, QString fileName);
  bool importModelsFromReport(RealSpace& parameterSpace, QString fileName, bool strict);
  virtual void reallocate();

  int _parameterCount;
  int _targetCount;
  int * _parameters;
  double * _misfits;

  QMutex _setMutex;
  QMutex _modelMutex;
};

} // namespace DinverCore

#endif // MODELSET_H
