/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-05-06
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef NEIGHBORHOOD_H
#define NEIGHBORHOOD_H

#include <QGpCoreTools.h>

#include "DinverCoreDLLExport.h"
#include "ForwardSignal.h"
#include "ActiveModel.h"

namespace DinverCore {

class GridModels;
class AbstractForward;
class VoronoiNavigator;
class ModelSet;
class ActiveModels;
class BestModels;
class UniqueRandom;
class ReportWriter;
class Parameter;
class Model;
class RealSpace;

class DINVERCORE_EXPORT Neighborhood
{
  TRANSLATIONS("Neighborhood")
public:
  Neighborhood();
  ~Neighborhood();

  bool operator==( const Neighborhood& o ) const;

  void setForward( AbstractForward * forward );
  void setStorage();
  bool setThreadCount( int nThreads );
  void setNr( int nr );
  void setGiveUp( double giveUp ) { _giveUp = giveUp; }
  void setSeed( int seed );
  bool openReport( const QString& fileName );
  void wake();
  void sleep();
  void timeReport() const;
  bool random( int ns = 50, int nw = 2 );
  bool optimization(int ns = 50, int nw = 2);

  bool importModels(QString fileName, bool strict = true);
  void clear();

  void lock() const;
  void unlock() const;

  int visitedModelCount() const;
  int validModelCount() const;
  int activeModelCount() const;
  int bestModelCount() const;
  int rejectedCount() const { return _rejectedCount; }
  int giveUpCount() const { return _giveUpCount; }
  int variableParameterCount () const;
  uint checksum() const { return _parameterSpaceChecksum; }

  double misfit( int modelIndex ) const;
  AbstractForward * forward() { return _forwards.isEmpty() ? 0 : _forwards.first(); }
  const AbstractForward * forward() const { return _forwards.isEmpty() ? 0 : _forwards.first(); }
  const Parameter * variableParameter( int paramIndex ) const;
  double variableParameterValue( int modelIndex, int paramIndex ) const;
  const int * model( int modelIndex ) const;
  int bestModelIndex() const;

  void terminate(bool t=true);
  void start( int itmax = 50, int ns0 = 50, int ns = 50, int nr = 50 );
private:
  void add(AbstractForward * forward);
  void stopForwards();
  bool terminated() { return _terminated.testAndSetOrdered(true,true); }
  bool optimizationEnd( bool exitCode );
  AbstractForward * nextForward();
  int removeDeadModels();
  double * scales();

  QVector<AbstractForward *> _forwards;

  ModelSet * _allModels;
  ActiveModels * _activeModels;
  BestModels * _bestModels;
  BestModels * _originalBestModels;
  UniqueRandom * _generator;

  ReportWriter * _report;
  uint _parameterSpaceChecksum;

  double _giveUp;

  int _deadCount;
  int _giveUpCount;
  int _rejectedCount;
  int _maxModelCount;

  int _timeAdd;
  int _timeMainWait;
  int _timeStartEnd;
  int _timeTotal;

  ForwardSignal _forwardSignal;
  QAtomicInt _terminated;
};

} // namespace DinverCore

#endif // NEIGHBORHOOD_H
