/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-15
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PARAMETER_H
#define PARAMETER_H

#include <QGpCoreTools.h>

#include "DinverCoreDLLExport.h"
#include "ParamCondition.h"
#include "ParameterGrid.h"

namespace DinverCore {

class DINVERCORE_EXPORT Parameter : public XMLClass
{
  TRANSLATIONS( "Param" )
public:
  Parameter();
  virtual ~Parameter();

  uint checksum() const;

  void setFussy( bool f ) { _fussy = f; }
  bool isFussy() const { return _fussy; }

  inline void getRectangularLimits( double& min, double& max ) const;
  inline void getLimits( double& min, double& max ) const;
  inline void getGridLimits( int& min, int& max ) const;
  inline void getGridLimits( double& min, double& max ) const;
  inline bool isOk() const;
  inline bool isOkDebug() const;
  void conditionDiagnostic() const;
  void humanInfo() const;

  void setScale( ParameterGrid::Scale s ) { _grid.setScale( s ); }
  void setPrecision( double p ) { _grid.setPrecision( p ); }
  void setMinimum( double m ) { _grid.setMinimum( m ); }
  double minimum() const { return _grid.minimum(); }
  void setMaximum( double m ) { _grid.setMaximum( m ); }
  double maximum() const { return _grid.maximum(); }
  bool initGrid() {return _grid.init();}
  bool isFixed() const { return _grid.minimum() >= _grid.maximum(); }

  void setRealValue( double val ) { _value=val; }
  bool setRealValue( QDataStream& s );
  double realValue() const { return _value; }
  int gridCount() const { return _grid.count(); }
  int gridValue() const { return _grid.gridValue( _value ); }
  void setGridValue( int val ) { _value = realValue( val ); }
  void setGridValue( double val ) { _value = realValue( val ); }
  double realValue( int val ) const { return _grid.realValue( val ); }
  double realValue( double val ) const { return _grid.realValue( val ); }
  int gridValue( double val ) const { return _grid.gridValue( val ); }

  void setName(QString name) {_name=name;}
  const QString& name() const {return _name;}

  void setUnit(QString unit) {_unit=unit;}
  const QString& unit() const {return _unit;}

  void addCondition( AbstractCondition * c );
private:
  QString _name;
  QString _unit;
  bool _fussy;
  QVector<ParamCondition> _conditions;
  ParameterGrid _grid;
  double _value;
protected:
  virtual const QString& xml_tagName() const {return xmlParamTag;}
  static const QString xmlParamTag;
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
};

inline void Parameter::getRectangularLimits(double& min, double& max) const
{
  min = minimum();
  max = maximum();
}

inline void Parameter::getLimits( double& min, double& max ) const
{
  for (QVector<ParamCondition>::const_iterator it = _conditions.begin(); it != _conditions.end(); ++it ) {
    it->getLimits( min, max );
  }
}

inline void Parameter::getGridLimits( int& min, int& max ) const
{
  double dmin, dmax;
  getRectangularLimits( dmin, dmax);
  getLimits( dmin, dmax );
  _grid.getLimits( min, max, dmin, dmax );
}

inline void Parameter::getGridLimits( double& min, double& max ) const
{
  double dmin, dmax;
  getRectangularLimits( dmin, dmax);
  getLimits( dmin, dmax );
  _grid.getLimits( min, max, dmin, dmax );
}

inline bool Parameter::isOk() const
{
  double min, max;
  getRectangularLimits(min, max);
  getLimits(min, max);
  if (isFixed()) {
    return (_value>=min && _value<=max);
  } else {
    return (_value>min && _value<max);
  }
}

inline bool Parameter::isOkDebug() const
{
  double min, max;
  getRectangularLimits(min, max);
  getLimits(min, max);
  if (isFixed()) {
    if (_value>=min && _value<=max) return true;
    printf("    %lg <= %lg <= %lg ? oh nooo!\n",min,_value,max);
  } else {
    if (_value>min && _value<max) return true;
    printf("    %lg < %lg < %lg ? oh nooo!\n",min,_value,max);
  }
  return false;
}

} // namespace DinverCore

#endif // PARAMETER_H
