/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-05-17
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PARAMETERGRID_H
#define PARAMETERGRID_H

#include <math.h>

#include "DinverCoreDLLExport.h"

namespace DinverCore {

class DINVERCORE_EXPORT ParameterGrid
{
public:
  ParameterGrid();
  ~ParameterGrid();

  enum Scale {Linear, Log};

  void setScale(Scale s) {_scale=s;}
  void setPrecision(double p) {ASSERT( p >= 0.0 ); _precision=p;}

  void setMinimum(double m) {_minimum=m;}
  double minimum() const {return _minimum;}

  void setMaximum(double m) {_maximum=m;}
  double maximum() const {return _maximum;}

  bool init();

  int count() const { return _n; }
  inline double realValue( int index ) const;
  inline double realValue( double index ) const;
  inline int gridValue( double val ) const;
  inline void getLimits( int& min, int& max, double dmin, double dmax ) const;
  inline void getLimits( double& min, double& max, double dmin, double dmax ) const;
private:
  void cacheRealLinearValues();
  void cacheRealLogValues();

  Scale _scale;
  double _minimum;
  double _maximum;
  double _precision;
  double * _realValues;
  int _n;
  double _a, _b, _c;
};

inline double ParameterGrid::realValue( int index ) const
{
  return _realValues[ index ];
}

inline double ParameterGrid::realValue( double index ) const
{
  switch( _scale) {
  case Linear:
    break;
  case Log:
    return _minimum*pow(_c, index);
  }
  return _minimum + index * _b;
}

inline int ParameterGrid::gridValue( double val ) const
{
  switch( _scale) {
  case Linear:
    break;
  case Log:
    return (int) round( log( val * _a ) * _b );
  }
  return (int) round( ( val - _minimum ) * _a );
}

inline void ParameterGrid::getLimits( int& min, int& max, double dmin, double dmax ) const
{
  switch (_scale) {
  case Linear:
    min = (int) ceil ( ( dmin - _minimum ) * _a + 1e-10 );
    max = (int) floor ( ( dmax - _minimum ) * _a - 1e-10 );
    break;
  case Log:
    min = (int) ceil( log( dmin * _a ) * _b + 1e-10 );
    max = (int) floor( log( dmax * _a ) * _b - 1e-10 );
    break;
  }
}

inline void ParameterGrid::getLimits( double& min, double& max, double dmin, double dmax ) const
{
  switch (_scale) {
  case Linear:
    min = ( dmin - _minimum ) * _a;
    max = ( dmax - _minimum ) * _a;
    break;
  case Log:
    min = log( dmin * _a ) * _b;
    max = log( dmax * _a ) * _b;
    break;
  }
}

} // namespace DinverCore

#endif // PARAMETERGRID_H
