/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2014-09-02
**  Copyright: 2014
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef QUADRATICCONDITION_H
#define QUADRATICCONDITION_H

#include <QGpCoreTools.h>

#include "AbstractCondition.h"
#include "DinverCoreDLLExport.h"

namespace DinverCore {

class DINVERCORE_EXPORT QuadraticCondition : public AbstractCondition
{
public:
  enum Type {LessThan, GreaterThan};

  QuadraticCondition(double coef1, Parameter * p1,
                     double coef2, Parameter * p2, Type type, double constant);
  QuadraticCondition(double coef1, Parameter * p1,
                     double coef2, Parameter * p2,
                     double coef3, Parameter * p3, Type type, double constant);
  ~QuadraticCondition();

  virtual bool operator==(const AbstractCondition& o) const;

  virtual const QString& xml_tagName() const {return xmlQuadraticConditionTag;}
  static const QString xmlQuadraticConditionTag;

  inline virtual void getLimits(int paramIndex, double& min, double& max) const;
  virtual bool adjustRanges();
  virtual void humanInfo() const;
protected:
  virtual uint internalChecksum() const;
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
private:
  Type _type;
  double * _coef;
  double _constant;
};

} // namespace DinverCore

#include "Parameter.h"

namespace DinverCore {

inline void QuadraticCondition::getLimits(int paramIndex, double& min, double& max) const
{
  TRACE;
  if(min<0.0) {
    App::stream() << tr("Quadratic condition cannot be used with negative parameters, skipping condition.") << endl;
    return;
  }
  double threshold=_constant;
  for(int i=count()-1; i>=0; i--) {
    if(i!=paramIndex) {
      double v=parameter(i).realValue();
      threshold-=_coef[i]*v*v;
    }
  }
  threshold=sqrt(threshold);
  switch (_type) {
  case LessThan:
    if(threshold<max) {
      max=threshold;
    }
    break;
  case GreaterThan:
    if(threshold>min) {
      min=threshold;
    }
    break;
  }
}

} // namespace DinverCore

#endif // QUADRATICCONDITION_H
