/***************************************************************************
**
**  This file is part of DinverCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-05-06
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef VORONOINAVIGATOR_H
#define VORONOINAVIGATOR_H

#include <QGpCoreTools.h>
#include "DinverCoreDLLExport.h"
#include "ScaledModels.h"
#include "PdfCurve.h"
#include "Model.h"

namespace DinverCore {

class DINVERCORE_EXPORT VoronoiNavigator
{
  TRANSLATIONS("VoronoiNavigator")
public:
  VoronoiNavigator(const ScaledModels * m);
  ~VoronoiNavigator();

  void cellLimits( int& xMin, int &xMax, int& iMin, int& iMax );
  PdfCurve intersections(double xMinLimit, double xMaxLimit);

  inline void setCurrentPoint( int index );
  inline void setCurrentPoint( const int * coordinates );

  int currentCell() const { return _currentCell; }
  QVector<int> cellAt(const int * p) const;
  QVector<int> cellAt(const double * p) const;
  int cellCount() const { return _cells->modelCount(); }

  void setCurrentAxis(int a);
  void incrementAxis();
  int currentAxis() const { return _currentAxis; }

  void setValue(int v) { _currentPoint[ _currentAxis ] = v * _cells->scale(_currentAxis); }
  void setValue(double v) { _currentPoint[ _currentAxis ] = v * _cells->scale(_currentAxis); }
  void setCurrentCell(int index) {_currentCell = index;}

  void checkAxisDistances();
  void printCurrentPoint();
  void checkCurrentPoint();
private:
  void setCell(int * cells, int nCells);
  void lowerIntersections(PdfCurve& f, double xMinLimit, int * cells, int nCells, double invS2);
  void higherIntersections(PdfCurve& f, double xMaxLimit, int * cells, int nCells, double invS2);

  const ScaledModels * _cells;

  int _currentAxis;
  int _currentCell;
  double * _currentPoint;

  double * _axisDistances;
};

inline void VoronoiNavigator::setCurrentPoint(int index)
{
  TRACE;
  _currentCell = index;
  int nd = _cells->parameterCount();
  for ( int id = 0; id < nd; id++ ) {
    _currentPoint[ id ] = _cells->v(id)[_currentCell];
  }
}

inline void VoronoiNavigator::setCurrentPoint(const int * coordinates)
{
  TRACE;
  _currentCell = -1;
  int nd = _cells->parameterCount();
  for ( int id = 0; id < nd; id++ ) {
    _currentPoint[ id ] = _cells->scale(id) * coordinates[id];
  }
}

} // namespace DinverCore

#endif // VORONOINAVIGATOR_H
