/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-18
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PARAMLAYER_H
#define PARAMLAYER_H

#include <DinverCore.h>
#include <QGpCoreTools.h>
#include <QGpCoreWave.h>

#include "DinverDCCoreDLLExport.h"

namespace DinverDCCore {

class ParamProfile;

class DINVERDCCORE_EXPORT ParamLayer : public XMLClass
{
  TRANSLATIONS( "ParamLayer" )
public:
  ParamLayer( const ParamLayer& o );
  ParamLayer( ParamProfile * profile, int index );
  ParamLayer( ParamProfile * profile, int index, double minValue, double maxValue, ParamProfile * linkedTo );
  ParamLayer( ParamProfile * profile, int index, double minValue, double maxValue, double minDepth, double maxDepth );

  virtual const QString& xml_tagName() const {return xmlParamLayerTag;}
  static const QString xmlParamLayerTag;

  enum Shape {Uniform, Linear, LinearIncrease, LinearDecrease, PowerLaw};

  ParamProfile * profile() const { return _profile; }
  void setProfile(ParamProfile * p) { _profile = p; }

  int index() const {return _index;}
  void setIndex(int i) {_index=i;}

  QString name() const;
  QString unit() const;
  static inline QString name(QString profileShortName, int index);

  Shape shape() const { return _shape; }
  void setShape( Shape s ) { _shape = s; }
  bool isUniformShape() { return _shape==Uniform; }

  int nSubLayers() const { return _nSubLayers; }
  void setNSubLayers( int n ) { _nSubLayers = n; }

  const QString& linkedTo() const { return _linkedTo; }
  void setLinkedTo( const QString& l ) { _linkedTo = l; }
  void setLinkedTo();
  void setPtrLink( ParamLayer * layer ) {_ptrLink=layer;}
  void setLinkedDepth(bool last, ParamLayer * prev, RealSpace& ps);
  bool isLinkedTo(ParamLayer * o);

  bool isDepth() const { return _isDepth; }
  void setDepth( bool d ) { _isDepth = d; }

  bool isLastParamCondition() const { return _lastParamCondition; }
  void setLastParamCondition( bool c ) { _lastParamCondition= c; }

  double topMinimumValue() const { return _topMin; }
  void setTopMinimumValue( double v ) { _topMin = v; }

  double topMaximumValue() const { return _topMax; }
  void setTopMaximumValue( double v ) { _topMax = v; }

  double bottomMinimumValue() const { return _bottomMin; }
  void setBottomMinimumValue( double v ) { _bottomMin = v; }

  double bottomMaximumValue() const { return _bottomMax; }
  void setBottomMaximumValue( double v ) { _bottomMax = v; }

  double minimumDepth() const { return _dhMin; }
  void setMinimumDepth( double v ) { _dhMin = v; }

  double maximumDepth() const { return _dhMax; }
  void setMaximumDepth( double v ) { _dhMax = v; }

  void setTopDepthIndex( int& index );
  int topDepthIndex() const { return _topDepthIndex; }

  double topDepth() const { return _finalTopDepth; }
  double bottomDepth() const { return _finalBottomDepth; }
  double fixedBottomDepth() const;
  void setFinalDepths(double& z);
  void setFinalProfileDepthsParam( Profile& profile, double & z );
  void setFinalProfileDepthsCondition( Profile& min, Profile& max, double & z );
  inline void setFinalProfileValuesParam( Profile& profile );
  inline void setFinalProfileValuesCondition( Profile& min, Profile& max );

  bool valueToParam(RealSpace& ps);
  bool depthToParam(bool last, ParamLayer * prev, RealSpace& ps);
  void collectValueParameters( QList<Parameter *>& params ) const;

  Parameter * topParam() const { return _topP; }
  Parameter * bottomParam() const { return _bottomP; }
  Parameter * dhParam() const { return _dhP; }
protected:
  double fixedThickness(ParamLayer *& ref);
  void setFinalProfileValues( Profile& profile, double topV, double bottomV );

  ParamProfile * _profile;
  int _index;
  Shape _shape;
  int _nSubLayers;
  double _topMin, _topMax, _bottomMin, _bottomMax;

  bool _lastParamCondition;

  QString _linkedTo;
  ParamLayer * _ptrLink;
  bool _isDepth;
  double _dhMin, _dhMax;

  Parameter * _topP, * _bottomP, * _dhP;
  int _topDepthIndex;
  double _finalTopDepth;
  double _finalBottomDepth;
protected:
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
};

inline QString ParamLayer::name(QString profileShortName, int index)
{
  TRACE;
  return profileShortName+QString::number(index);
}

inline void ParamLayer::setFinalProfileValuesParam(Profile& profile)
{
  TRACE;
  setFinalProfileValues( profile, _topP->realValue(), _bottomP->realValue() );
}

inline void ParamLayer::setFinalProfileValuesCondition( Profile& min, Profile& max )
{
  TRACE;
  setFinalProfileValues( min, _topMin, _bottomMin );
  setFinalProfileValues( max, _topMax, _bottomMax );
}


} // namespace DinverDCCore

#endif // PARAMLAYER_H
