/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-18
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ParamProfile_H
#define ParamProfile_H

#include <stdio.h>

#include <DinverCore.h>
#include <QGpCoreTools.h>
#include <QGpCoreWave.h>

#include "DinverDCCoreDLLExport.h"

namespace DinverDCCore {

class ParamLayer;

typedef QList<ParamLayer *> LayerList;
typedef QMap<QString,ParamLayer *> LayerMap;

#define _minRaw _raw
#define _minResampled _resampled

class DINVERDCCORE_EXPORT ParamProfile  : public XMLClass
{
  TRANSLATIONS( "ParamProfile" )
public:
  enum Type { Param, Condition };

  ParamProfile();
  ParamProfile( const ParamProfile& o );
  ParamProfile( QString shortName, QString longName, QString unit,
                   double defaultMinimum, double defaultMaximum,
                   Type type, SimpleCondition::Type defaultCondition);
  virtual ~ParamProfile();

  Type type() const { return _type; }
  const QString& longName() const { return _longName; }
  const QString& shortName() const { return _shortName; }
  const QString& unit() const { return _unit; }
  double defaultMinimum() const { return _defaultMinimum; }
  double defaultMaximum() const { return _defaultMaximum; }
  SimpleCondition::Type defaultCondition() const { return _defaultCondition; }

  void insertLayer(int at, ParamLayer * layer);
  void addLayer( ParamLayer * layer ) { _layers.append(layer); }
  ParamLayer * layer( int index ) const { return _layers.at(index); }

  void collectDepthLinks(LayerMap& links);
  void setDepthLinks(const LayerMap& links);
  void setDepthLinkNames();

  bool toParam(RealSpace& ps, LayerMap& links);
  void setLinkedDepth(RealSpace& ps);
  int nLayers() const {return _layers.count();}
  void initFinalProfile();
  void setFinalDepths(int fromLayer = 0);
  void setFinalProfileDepths(int fromLayer = 0);
  void setFinalProfileFrom(int layer = 0);
  void setFinalProfileAt(int layer);
  void collectDepths(QVector<double>& depths) const {_raw.collectDepths(depths);}
  const QVector<double>& depths() const {return _resampled.depths();}
  const QVector<double>& values() const {return _resampled.values();}
  const QVector<double>& minValues() const {return _minResampled.values();}
  const QVector<double>& maxValues() const {return _maxResampled.values();}
  inline void resampleProfile(QVector<double>& baseD);
  void writeReport(QDataStream& s) const {_raw.writeReport(s);}
  void collectValueParameters( QList<Parameter *>& params ) const;
  void pRaw() const;
  void pResampled() const;
  void pMaxRaw() const;
  void pMaxResampled() const;
protected:
  QString _longName;
  QString _shortName;
  QString _unit;
  double _defaultMinimum, _defaultMaximum;
  SimpleCondition::Type _defaultCondition;
  Type _type;

  LayerList _layers;

  Profile _raw;
  Profile _resampled;
  Profile _maxRaw;
  Profile _maxResampled;
protected:
  virtual const QString& xml_tagName() const {return xmlParamProfileTag;}
  static const QString xmlParamProfileTag;
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
};

inline void ParamProfile::resampleProfile(QVector<double>& baseD)
{
  TRACE;
  _resampled.resample(_raw, baseD);
  if (_type == Condition) _maxResampled.resample(_maxRaw, baseD);
}

} // namespace DinverDCCore

#endif // PARAMPROFILE_H
