/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-21
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef TargetList_H
#define TargetList_H

#include <QGpCoreTools.h>
#include <QGpCoreWave.h>
#include <DinverCore.h>

#include "DCReportBlock.h"
#include "PoissonCondition.h"
#include "ModalCurveTarget.h"
#include "AutocorrTarget.h"
#include "ValueTarget.h"
#include "RefractionTarget.h"
#include "DinverDCCoreDLLExport.h"

namespace DinverDCCore {

class ParamGroundModel;
class ParamProfile;

class DINVERDCCORE_EXPORT TargetList : public XMLClass
{
  TRANSLATIONS("TargetList")
public:
  TargetList();
  TargetList(const TargetList& o);
  virtual ~TargetList();

  virtual const QString& xml_tagName() const {return xmlTargetListTag;}
  static const QString xmlTargetListTag;

  void setParamProfiles(ParamGroundModel * gm, RealSpace& ps);
  inline bool isPoissonRatioOk( const GroundParameter * from );
  double misfit( int nDimensions, bool& ok );
  void surfaceMisfit( double& totalMisfit, double& totalWeight,
                      LayeredModel * surfModel,
                      int nDimensions, bool& ok );
  void refractionMisfit( double& totalMisfit, double& totalWeight,
                         LayeredModel * surfModel, const QVector<double>& pitch,
                         int nDimensions, bool& ok );
  void refractionMisfit( double& totalMisfit, double& totalWeight,
                         TiltLayeredModel * tiltModel,
                         int nDimensions, bool& ok );
  inline void writeReport(ReportWriter * report) const;

  ModalCurveTarget& dispersionTarget() {return _dispersionTarget;}
  const ModalCurveTarget& dispersionTarget() const {return _dispersionTarget;}

  AutocorrTarget& autocorrTarget() {return _autocorrTarget;}
  const AutocorrTarget& autocorrTarget() const {return _autocorrTarget;}

  ModalCurveTarget& ellipticityCurveTarget() {return _ellipticityCurveTarget;}
  const ModalCurveTarget& ellipticityCurveTarget() const {return _ellipticityCurveTarget;}

  ValueTarget& ellipticityPeakTarget() {return _ellipticityPeakTarget;}
  const ValueTarget& ellipticityPeakTarget() const {return _ellipticityPeakTarget;}

  RefractionTarget& refractionVpTarget() {return _refractionVpTarget;}
  const RefractionTarget& refractionVpTarget() const {return _refractionVpTarget;}

  RefractionTarget& refractionVsTarget() {return _refractionVsTarget;}
  const RefractionTarget& refractionVsTarget() const {return _refractionVsTarget;}

  void setMisfitType(StatValue::MisfitType t);

  void validateTargets();
  int dispersionSampleCount() const;
protected:
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual void xml_polishChild( XML_POLISHCHILD_ARGS );
  virtual void xml_polish(XML_POLISH_ARGS);
private:
  friend class DCReportBlock;

  void initFactories();
  void initProfiles();
  void deleteFactories();
  double curveMisfit(const QList<ModalCurve>& curveList, ModalFactory& f,
                     const Target& target, int nDimensions);
  double curveMisfit(const QList<RefractionCurve>& curveList, RefractionFactory& f,
                     const Target& target, int nDimensions);
  double logMisfit(const QList<ModalCurve>& curveList, ModalFactory& f,
                   const Target& target, int nDimensions);
  bool surfaceWave() const {return !_dispersionTarget.curves().isEmpty() ||
                                   !_autocorrTarget.curves().isEmpty() ||
                                   !_ellipticityCurveTarget.curves().isEmpty() ||
                                   _ellipticityPeakTarget.value().isValid(); }
  // Targets
  ModalCurveTarget _dispersionTarget;
  AutocorrTarget _autocorrTarget;
  ModalCurveTarget _ellipticityCurveTarget;
  ValueTarget _ellipticityPeakTarget;
  RefractionTarget _refractionVpTarget;
  RefractionTarget _refractionVsTarget;

  // Computation storages
  DispersionFactory * _dispersionFactory;
  AutocorrFactory * _autocorrFactory;
  EllipticityFactory * _ellipticityFactory;
  RefractionFactory * _refractionVpFactory;
  RefractionFactory * _refractionVsFactory;

  // Ground structure
  ParamProfile * _vp, * _vs, * _rho, * _pitch;
  PoissonCondition * _poissonCondition;

  LayeredModel * surfaceWaveModel();
};

inline void TargetList::writeReport(ReportWriter * report) const
{
  TRACE;
  DCReportBlock::write(report, *this);
}

inline bool TargetList::isPoissonRatioOk( const GroundParameter * from )
{
  TRACE;
  if (_poissonCondition)
    return _poissonCondition->isOk(from);
  else
    return true;
}

} // namespace DinverDCCore

#endif // TARGETLIST_H
