/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-04-11
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef DFFTPACKCACHE_H
#define DFFTPACKCACHE_H

#include <QGpCoreTools.h>

namespace GeopsyCore {

#ifndef NO_FFT

class DFFTPackKey : public AbstractNumericalKey
{
  DECLARE_NUMERICALKEY(DFFTPackKey)
public:
  enum Number { Real=0, Complex=1 };

  DFFTPackKey( int nSamples, Number number ) {
    _nSamples = nSamples;
    _number = number;
  }
  virtual bool operator==( const AbstractNumericalKey& o ) {
    const DFFTPackKey& mo = static_cast<const DFFTPackKey&>(o);
    return _nSamples == mo._nSamples && _number == mo._number;
  }
  virtual int hash() const { return _nSamples*(_number*2-1); }
  inline virtual AbstractNumericalCache * createCache();
  inline virtual int byteCount() const;

  int nSamples() const { return _nSamples; }
  Number number() const { return _number; }
private:
  int _nSamples;
  Number _number;
};

class DFFTPackCache : public AbstractNumericalCache
{
public:
  DFFTPackCache( DFFTPackKey * key );
  ~DFFTPackCache();

  static inline const DFFTPackCache * begin( int nSamples, DFFTPackKey::Number number );
  virtual void init();

  double * wsave() const { return _wsave; }
private:
  double * _wsave;
};

inline const DFFTPackCache * DFFTPackCache::begin( int nSamples, DFFTPackKey::Number number )
{
  return static_cast<const DFFTPackCache *>(
      AbstractNumericalCache::begin( new DFFTPackKey( nSamples, number ) ) );
}

inline AbstractNumericalCache * DFFTPackKey::createCache( )
{
  return new DFFTPackCache( this );
}

inline int DFFTPackKey::byteCount() const
{
  switch (_number) {
  case Real:
    break;
  case Complex:
    return (2*_nSamples+15)*sizeof(double);
  }
  return (_nSamples+15)*sizeof(double);
}

#endif // NO_FFT

} // namespace GeopsyCore

#endif // DFFTPACKCACHE_H
