/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-11-22
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef DOUBLESIGNAL_H
#define DOUBLESIGNAL_H

#include "GeopsyCoreDLLExport.h"
#include "SignalTemplate.h"
#include "ComplexSignal.h"
#include "WindowingParameters.h"

namespace GeopsyCore {

class KeepSignal;
class MorletParameters;

class GEOPSYCORE_EXPORT DoubleSignal: public SignalTemplate<double>
{
   TRANSLATIONS("DoubleSignal")
public:
  enum SignalType {
    Waveform = 0,
    Spectrum = 1,
    Phase = 2,
    RealSpectrum = 3,   // For real spectrum and their fft's
    CAmpWaveform = 4,
    CPhaseWaveform = 5,
    ArrivalTime = 6,
    UndefinedSignalType = 7
  };

  DoubleSignal( );
  DoubleSignal( int n );
  DoubleSignal( const DoubleSignal& o );

  void lockProperties() const { _propertiesMutex.lockForWrite(); }
  void constLockProperties() const { _propertiesMutex.lockForRead(); }
  void unlockProperies() const { _propertiesMutex.unlock(); }

  // Properties
  double deltaT() const { return _deltaT; }
  double deltaF() const { return _deltaT; }
  virtual void setDeltaT( double newval ) { _deltaT = newval; }
  void setDeltaF( double newval ) { setDeltaT(newval); }
  double samplingFrequency() const { return 1.0/_deltaT; }
  double duration() const { return _deltaT * _nSamples; }
  inline virtual void setNSamples( int n );
  SignalType type() const { return _type; }
  void setType( SignalType t ) { if( t!=UndefinedSignalType ) _type = t; }
  inline bool isReal() const;
  inline bool isComplex() const;
  inline bool isSpectrum() const;
  void copyBasicProperties( const DoubleSignal& o );

  static SignalType type( QString t );
  static QString typeString( SignalType st );

  inline void copySamplesFrom( const DoubleSignal * sig );
  void copySamplesFrom( const DoubleSignal * sig, double sigStart, double thisStart, double tToCopy );
  void copySamplesFrom( const DoubleSignal * sig, int isigStart, int ithisStart, int nToCopy );
  void copyAmplitudeFrom( const DoubleSignal * sig );
  int add( const DoubleSignal * sig, double sigStart, double thisStart,
           double tToCopy, bool checkInvalid = false, double invalidValue = 1e99 );
  int add( const DoubleSignal * sig );

  // Funtion to perform signal processing
  void fastFourierTransform( SignalType st );

  SignalType saveType( SignalType st );
  void restoreType( SignalType st );
  const DoubleSignal * saveType( SignalType st ) const;
  void restoreType( const DoubleSignal * sig ) const;

  bool subtractValue( double val = 0.0 );
  bool filter( const FilterParameters& param );
  bool whiten();
  bool stddevClip( double factor );
  bool agc( double width );
  bool taper( double tmin, double tmax, double width );
  bool taper( int imin, int imax, int iwidth );
  bool abs();
  bool square();
  bool sqrt();
  bool decimateAmplitude( double delta );
  bool shift( double dt );
  bool unglitch( double threshold );
  bool smooth( const SmoothingParameters& param, double minFreq, double maxFreq );
  bool setValue( double val );

  bool discreteFourierTransform();
  bool overSample( double factor, const DoubleSignal * sig );
  bool decimateTime( int factor, const DoubleSignal * sig );
  bool hv( const DoubleSignal * zSig,  const DoubleSignal * hSig );
  bool correlation( const DoubleSignal * s1, const DoubleSignal * s2, double maxDelay );

  ComplexSignal * morletWavelet( const MorletParameters& param ) const;
  ComplexSignal * positiveComplexSpectrum( bool fullSize = false ) const;

  void stalta( double tsta, double tlta, DoubleSignal * ratio ) const;
  void staltaToKeep( const WindowingParameters::SLtaParam& param, KeepSignal * keep, double delay ) const;
  void clipMaxToKeep( double percentage, KeepSignal * keep, double delay ) const;
  double correlation( const DoubleSignal * sig, double sigStart, double thisStart, double length ) const;

  double amplitudeAt( double abscissa ) const;
  double interpoleAt( double abscissa ) const;
  double maximumAmplitude( int itmin, int itmax ) const;
  double averageAmplitude( int itmin, int itmax ) const;
  double energy( double minF, double maxF) const;

  inline Complex complex( const double * samples, int i ) const;
  inline double amplitude2( const double * samples, int i ) const;
  double amplitude( const double * samples, int i ) const;
  void setAmplitude( double * samples, int i, double val ) const;
  inline double phase( const double * samples, int i ) const;
  inline double re( const double * samples, int i ) const;
  inline double im( const double * samples, int i ) const;

  void debugPrint() const;
protected:
  mutable QReadWriteLock _propertiesMutex;
  double _deltaT;
  SignalType _type;
  int _nyquistIndex;
private:
  bool smoothFixedTriang( double width, double minF, double maxF );
  bool smoothFreqDepTriang( double ratio, double minF, double maxF );
  bool smoothKonno( double constant, double minF, double maxF );
  static void filb3( double fc, int ak, double pas, double *a, double *b );
  static void frecur( int la, double *a, double *xa, int lb, double *b, double *yb,
                      int ndeb, int nfin, int npas, int isens, double *x, double *y );
  static void filrec( double fc, int type, int ndeb, int nfin, int npas, double *x,
                      double *y, int nfil, double pas );
  int commonSamples( double delay, int otherNSamples, int& thisStart, int& otherStart ) const;
};

inline void DoubleSignal::setNSamples( int n )
{
  TRACE;
  SignalTemplate<double>::setNSamples(n);
  _nyquistIndex = _nSamples >> 1;
}

inline double DoubleSignal::re( const double * samples, int i ) const
{
  TRACE;
  if ( i <= _nyquistIndex ) {
    return samples[i];
  } else {
    return samples[_nSamples-i];
  }
}

inline double DoubleSignal::im( const double * samples, int i ) const
{
  TRACE;
  if ( i == 0 ) {
    return 0.0;
  } else if ( i < _nyquistIndex ) {
    return samples[ _nSamples -i ];
  } else if ( i == _nyquistIndex && !( _nSamples & 0x00000001 ) ) {
    return 0.0; // For even sequency, sample at nyquist frequency
  } else {
    return -samples[ i ];
  }
}

inline Complex DoubleSignal::complex( const double * samples, int i ) const
{
  TRACE;
  if ( i == 0 ) {
    return samples[ i ];
  } else if ( i < _nyquistIndex ) {
    return Complex(samples[ i ], samples[ _nSamples - i ] );
  } else if ( i == _nyquistIndex && !( _nSamples & 0x00000001 ) ) {
    return samples[ i ]; // For even sequency, sample at nyquist frequency
  } else {
    return Complex(samples[ _nSamples - i ], samples[ i ] );
  }
}

inline double DoubleSignal::amplitude2( const double * samples, int i ) const
{
  TRACE;
  if ( i == 0 ) {
    return samples[ i ]*samples[ i ];
  } else if ( i != _nyquistIndex || ( _nSamples & 0x00000001 )  ) {
    return samples[ i ] * samples[ i ] +
           samples[ _nSamples - i ] * samples[ _nSamples - i ];
  } else { // For even sequency, sample at nyquist frequency
    return samples[ i ]*samples[ i ];
  }
}

inline double DoubleSignal::amplitude( const double * samples, int i ) const
{
  return ::sqrt( amplitude2( samples, i ) );
}

inline double DoubleSignal::phase( const double * samples, int i ) const
{
  return complex( samples, i ).phase();
}

inline void DoubleSignal::setAmplitude( double * samples, int i, double val ) const
{
  if ( i == 0 ) {
    samples[ i ] = val;
  } else if ( i != _nyquistIndex || ( _nSamples & 0x00000001 )  ) {
    double ampl = ::sqrt( samples[ i ] * samples[ i ] +
                          samples[ _nSamples - i ] * samples[ _nSamples - i ] );
    if( ampl == 0.0 ) return;
    double factor = val / ampl;
    samples[ i ] *= factor;
    samples[ _nSamples - i ] *= factor;
  }  else { // For even sequency, sample at nyquist frequency
    samples[ i ] = val;
  }
}

inline bool DoubleSignal::isComplex() const
{
  TRACE;
  switch ( _type ) {
  case Spectrum:
  case Phase:
  case CAmpWaveform:
  case CPhaseWaveform:
    return true;
  default:
    return false;
  }
}

inline bool DoubleSignal::isReal() const
{
  TRACE;
  switch ( _type ) {
  case Waveform:
  case RealSpectrum:
    return true;
  default:
    return false;
  }
}

inline bool DoubleSignal::isSpectrum() const
{
  TRACE;
  switch ( _type ) {
  case Spectrum:
  case Phase:
  case RealSpectrum:
    return true;
  default:
    return false;
  }
}

inline void DoubleSignal::copySamplesFrom( const DoubleSignal * sig )
{
  SignalTemplate<double>::copySamplesFrom( sig );
}

} // namespace GeopsyCore

#endif // DOUBLESIGNAL_H
