/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-04-19
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef RELATIVETIMERANGE_H
#define RELATIVETIMERANGE_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"
#include "TimeRange.h"

namespace GeopsyCore {

class Signal;
class StationSignals;

class GEOPSYCORE_EXPORT RelativeTimeRange : public TimeRange, public XMLClass
{
  TRANSLATIONS("RelativeTimeRange")
public:
  RelativeTimeRange();
  RelativeTimeRange( const RelativeTimeRange& o ) : TimeRange(o), XMLClass() { operator=(o); }

  inline void operator=( const RelativeTimeRange& o );

  enum Type { Absolute, Delta, Signal, Pick };

  void setStartType( Type t ) { _startType = t; }
  Type startType() const { return _startType; }

  void setEndType( Type t ) { _endType = t; }
  Type endType() const { return _endType; }

  QString startTypeString() const { return type2String(_startType); }
  void setStartType( QString t ) { setStartType(string2Type(t)); }

  QString endTypeString() const { return type2String(_endType); }
  void setEndType( QString t ) { setEndType(string2Type(t)); }

  int startPick() const { return _startPick; }
  void setStartPick( int p ) { if(p>=0) _startPick = p; }

  int endPick() const { return _endPick; }
  void setEndPick( int p ) { if(p>=0) _endPick = p; }

  double deltaT() const { return _deltaT; }
  void setDeltaT( double dt ) { _deltaT = dt; setEnd( start()+_deltaT ); }

  bool useFirstSignal() const { return _useFirstSignal; }
  void setUseFirstSignal( bool f ) { _useFirstSignal = f; }

  TimeRange absoluteRange() const;
  TimeRange absoluteRange(const GeopsyCore::Signal * sig) const;
  TimeRange absoluteRange(const StationSignals * sigs) const;

  virtual void xml_attributes( XML_ATTRIBUTES_ARGS ) const;
  virtual bool xml_setAttributes( XML_SETATTRIBUTES_ARGS );

  virtual const QString& xml_tagName() const { return xmlRelativeTimeRangeTag;}
  static const QString xmlRelativeTimeRangeTag;
private:
  static QString type2String( Type t );
  static Type string2Type( QString t );

  double absoluteStart(const GeopsyCore::Signal * sig) const;
  double absoluteEnd(const GeopsyCore::Signal * sig) const;

  int _startPick,_endPick;
  double _deltaT;
  Type _startType:2;
  Type _endType:2;
  bool _useFirstSignal:1;
};

inline void RelativeTimeRange::operator=( const RelativeTimeRange& o )
{
  TRACE;
  _startPick = o._startPick;
  _endPick = o._endPick;
  _deltaT = o._deltaT;
  _startType = o._startType;
  _endType = o._endType;
  _useFirstSignal = o._useFirstSignal;
  TimeRange::operator=( o );
}

} // namespace GeopsyCore

// Allow passing RelativeTimeRange as a QVariant
using namespace GeopsyCore;
Q_DECLARE_METATYPE(RelativeTimeRange);

#endif // RELATIVETIMERANGE_H
