/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-11-10
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SIGNAL_H
#define SIGNAL_H

#include <QGpCoreTools.h>
#include "GeopsyCoreDLLExport.h"
#include "SignalFile.h"
#include "DoubleSignal.h"
#include "SignalMember.h"
#include "SparseTimeRange.h"
#include "MiniSeedRecords.h"
#include "GuralpRecords.h"

namespace GeopsyCore {

class SubSignalPool;
class SignalDB;

#define SIGNAL_MEMBER_DECL(type, name, id) \
  bool is##id##MemberDefined() const { return _optionalMembers.contains( id ); } \
  void remove##id##Member() { delete _optionalMembers.take(id); } \
  type& name##Member() \
  { \
    QMap<Data, SignalMember *>::iterator it; \
    it=_optionalMembers.find( id ); \
    if (it!=_optionalMembers.end()) \
      return static_cast<type&>(*it.value()); \
    else \
      return static_cast<type&>(*_optionalMembers.insert(id, new type).value()); \
  } \
  const type& name##Member() const \
  { \
    QMap<Data, SignalMember *>::const_iterator it; \
    it=_optionalMembers.find( id ); \
    if (it!=_optionalMembers.end()) \
      return static_cast<const type&>(*it.value()); \
    else \
      return name##Default; \
  } \
  static type name##Default;
#define SIGNAL_MEMBER_IMP(type, name) \
  type Signal::name##Default;

class GEOPSYCORE_EXPORT Signal: public DoubleSignal, public SharedObject, public XMLClass
{
  TRANSLATIONS( "Signal" )
public:
  enum Data {
    UnknownData = 0,
    ID,
    T0,
    DeltaT,
    NSamples,
    ReceiverX,
    ReceiverY,
    ReceiverZ,
    SourceX,
    SourceY,
    SourceZ,
    Name,
    Type,
    TimePick,
    FileNumber,
    NumberInFile,
    DuplicateRays,
    DupID,
    DupAverage,
    DupStdErr,
    TimeReference,
    FileName,
    Duration,
    EndTime,
    Component,
    MaximumAmplitude,
    AverageAmplitude,
    IsOriginalFile,
    SampFreq,
    Pointer,
    CountPerVolt,
    VoltPerCount,
    VoltPerUnit,
    UnitPerVolt,
    CountPerUnit,
    UnitPerCount,
    AmplitudeUnit,
    SampleSize,
    Comments,
    HeaderModified,
    SourceReceiverDistance,
    SourceReceiverAzimuth,
    SourceReceiverRoughAzimuth, // Used for sort criteria else same as Azimuth
    ShortFileName,
    FileFormat,
    MiniSeedRecordList,
    GuralpRecordList,
    DataCount // This item should be the last one (= number of data)
  };
  class DataIndex {
  public:
    DataIndex(Data id=UnknownData, int index=0) {_id=id; _index=index;}
    Data _id;
    int _index;
  };
  class SortKey: public DataIndex {
  public:
    SortKey(DataIndex d, bool r=false) : DataIndex(d) {_reversed=r;}
    SortKey(Data d, int i=0, bool r=false) : DataIndex(d, i) {_reversed=r;}
    bool _reversed;
  };
  enum Components { Vertical, North, East, All, Horizontal, UndefinedComponent };
  enum PickWhat {Min = 1, Max = 2};
  enum AmplitudeUnits { UndefinedUnit, Displacement, Velocity, Acceleration, CustomUnit };

  Signal( SignalDB * db );
  Signal( const Signal& sig, SignalDB * db = 0 );
  virtual ~Signal();

  virtual const QString& xml_tagName() const {return xmlSignalTag;}
  static const QString xmlSignalTag;

  static Signal * newCopy( Signal * sig );

  void setTemporary();
  void copyBasicProperties( const Signal& p );

  bool operator<( const Signal& o ) const { return compare( o ) < 0;}
  bool operator>( const Signal& o ) const { return compare( o ) > 0;}
  bool operator==( const Signal& o ) const { return compare( o ) == 0;}
  int compare( const Signal& o ) const;

  // Modification control
  bool isHeaderModified() const { return _isHeaderModified;}
  void setHeaderModified( bool m ) {_isHeaderModified = m;}
  inline bool warnReadOnlySamples() const;
  bool isReadOnlySamples() const { return _isReadOnlySamples;}
  void setReadOnlySamples( bool ro ) { _isReadOnlySamples = ro;}
  void resetAmplitudes() const { _maxAmplitude = 1e99; _averageAmplitude = 1e99; }

  // Functions to save signal values into a stream
  bool writeSeg2( QFile& f ) const;
  bool writeAscii( QFile& f, int numInFile ) const;
  bool writeSac(QDataStream& s) const;
  bool writeSu(QDataStream& s, int indexInFile) const;
  bool writeGse( QFile& f ) const;
  bool writeMiniSeed( QFile& f ) const;

  // Interface functions to data members
  QString name() const { return _name;}
  void setName( QString n ) { _name = n; }
  Components component() const { return _component; }
  void setComponent( Components c ) { _component = c; }
  static Components userComponent( QString c );
  static Components standardComponent( QString c );
  static QString componentLetter(Components c);
  QString componentUserName() const { return userName( _component ); }
  QString componentStandardName() const { return standardName( _component ); }
  QString nameComponent() const;
  inline virtual QString debugName() const;

  const Point& source() const {return _source;}
  void setSource( const Point& p ) {_source = p;}
  const Point& receiver() const {return _receiver;}
  void setReceiver( const Point& p ) {_receiver=p;}
  double sourceReceiverDistance() const {return _source.distanceTo(_receiver);}
  double sourceReceiverAzimuth() const {return _source.azimuthTo(_receiver);}

  virtual void setNSamples( int n );
  virtual void setDeltaT( double newval );

  double t0() const { return _t0;}
  void setT0( double t );
  TIME t0AsTIME() const { return DateTime::capAddSecs( _timeReference, t0() ); }
  QDateTime timeReference() const { return _timeReference; }
  QString timeReferenceString() const { return _timeReference.toString( "dd/MM/yyyy hh:mm:ss" ); }
  void setTimeReference( QDateTime t ) { _timeReference = t; }
  void setTimeReference( QString t ) { _timeReference = QDateTime::fromString( t, "dd/MM/yyyy hh:mm:ss" ); }
  double endTime() const { return t0() + deltaT() * _nSamples; }
  const SparseTimeRange& timeRange() const { return _timeRange; }
  void setTimeRange(const SparseTimeRange& r);
  TIME endTimeAsTIME() const { return DateTime::capAddSecs( _timeReference, endTime() ); }

  double timePick( int index ) const { return timePickMember().value(index); }
  void setTimePick( int index, double time ) { timePickMember().setValue(index, time); }
  int timePickCount() const { return timePickMember().count(); }

  SignalFile * file() const { return _file; }
  void setFile( SignalFile * f ) { _file = f; }
  bool isOriginalFile() const { if ( file() && file() ->isOriginalFile() ) return true; else return false; }

  int numberInFile() const { return _numberInFile; }
  void setNumberInFile( int i ) { _numberInFile = i; }
  int offsetInFile() const { return _offsetInFile; }
  void setOffsetInFile( int o ) {_offsetInFile = o; }
  int byteIncrement() const { return _byteIncrement; }
  void setByteIncrement( int i ) { _byteIncrement = i; }
  void setMiniSeedRecords(const MiniSeedRecords& r) { miniSeedRecordsMember() = r; }
  const MiniSeedRecords& miniSeedRecords() const { return miniSeedRecordsMember(); }
  void setGuralpRecords(const GuralpRecords& r) {guralpRecordsMember()=r;}
  const GuralpRecords& guralpRecords() const {return guralpRecordsMember();}

  double countPerVolt() const { return _countPerVolt; }
  void setCountPerVolt( double c );

  double voltPerCount() const { return 1.0 / _countPerVolt; }
  void setVoltPerCount( double c ) { setCountPerVolt( 1.0/c ); }

  double voltPerUnit() const { return _voltPerUnit; }
  void setVoltPerUnit( double c );

  double unitPerVolt() const { return 1.0 / _voltPerUnit; }
  void setUnitPerVolt( double c ) { setVoltPerUnit( 1.0/c ); }

  double unitPerCount() const { return 1.0 / countPerUnit(); }
  double countPerUnit() const { return _countPerVolt * _voltPerUnit; }

  int id() const { return _id; }
  void setId( int id );

  SignalDB * database() const { return _db; }
  void setDatabase( SignalDB * db ) { _db = db; }

  AmplitudeUnits amplitudeUnit() const { return _amplitudeUnit; }
  void setAmplitudeUnit(AmplitudeUnits u) { _amplitudeUnit = u; }
  static AmplitudeUnits userAmplitudeUnit(QString u);
  static AmplitudeUnits standardAmplitudeUnit(QString u);
  QString amplitudeUnitUserName() const { return userName( _amplitudeUnit ); }
  QString amplitudeUnitStandardName() const { return standardName( _amplitudeUnit ); }
  QString effectiveAmplitudeUnit() const;

  static QString userName( Components c );
  static QString standardName( Components c );
  static QString userName( AmplitudeUnits u );
  static QString standardName( AmplitudeUnits u );

  double maximumAmplitude( int itmin=0, int itmax=0 ) const;
  double averageAmplitude( int itmin=0, int itmax=0 ) const;
  inline double amplitudeAt( double abscissa ) const;

  double duplicateRaysStdErr( int index ) const;
  double duplicateRaysAverage( int index ) const;
  int duplicateRaysID() const { return duplicateRaysMember().id(); }
  void setDuplicateRaysID( int newID ) {duplicateRaysMember().setId(newID); }
  SubSignalPool* duplicateRaysPool() const { return duplicateRaysMember().rays(); }
  void setDuplicateRaysPool( SubSignalPool * newPool ) {duplicateRaysMember().setRays(newPool); }

  QString comments() const { return commentsMember().value(); }
  void setComments(QString c) { commentsMember().setValue(c); }

  QVariant header( DataIndex d );
  bool setHeader( DataIndex d, QVariant val );
  static bool isReadOnlyData( DataIndex id );
  static bool isStoredData( DataIndex id );

  // Funtion to perform signal processing
  inline bool taper( const TimeRange& tw, double width );
  void copySamplesFrom( const Signal * sig ) { DoubleSignal::copySamplesFrom( sig ); }
  inline void copySamplesFrom( const Signal * sig, const TimeRange& tw );
  inline int add( const Signal * sig, const TimeRange& tw, bool checkInvalid = false, double invalidValue = 1e99 );
  void pickCoppens( int pickno, double delta, double end, double start = 0.,
                    bool relative = false, double mute = 0. );
  void pickMinMax( int pickno, double from, double to, PickWhat what );
  double correlation( const Signal * sig, const TimeRange& tw );
  void correlation( const Signal * s1, const Signal * s2, double maxDelay );

  // Function to restore old databases (compatibility)
  bool read( FILE *f, SignalDB * db);

  virtual double * lockSamples();
  virtual const double * constLockSamples() const;
#ifdef LOCK_SAMPLES_DEBUG
  double * lockSamples(char * file, int line) { return DoubleSignal::lockSamples(file, line); }
  const double * constLockSamples(char * file, int line) const { return DoubleSignal::constLockSamples(file, line); }
#endif

  static inline bool lessThan( const Signal * sig1, const Signal * sig2 );
  static void clearSortKeys() { _keys.clear();}
  static void addSortKey( Data d, int index=0, bool rev=false) { _keys.append( SortKey(d, index, rev) ); }
  static void addSortKey( DataIndex d, bool rev=false) { _keys.append( SortKey(d, rev) ); }
  static int sortKeyCount() { return _keys.count(); }
  static const SortKey& sortKey(int index) { return _keys.at(index); }

  static Components globalSeismographicNetworkComponent(const char * c);

  virtual void xml_writeProperties( XML_WRITEPROPERTIES_ARGS ) const;
  virtual void xml_writeChildren( XML_WRITECHILDREN_ARGS ) const;
  virtual bool xml_setProperty( XML_SETPROPERTY_ARGS );
  virtual XMLMember xml_member( XML_MEMBER_ARGS );
protected:
  // Updated by maxAmplitude()
  mutable double _maxAmplitude;
  // Updated by averageAmplitude()
  mutable double _averageAmplitude;
  mutable QMutex _amplitudeMutex;

  uint _isHeaderModified:1;
  uint _isReadOnlySamples:1;
  uint _unused:30;

  double _t0;
  SparseTimeRange _timeRange;
private:
  int _id;    // Unique ID in the database
  QDateTime _timeReference;  // reference time and date of acquisition
  Point _receiver;
  Point _source;
  QString _name;
  SignalFile * _file;
  int _numberInFile;
  qint64 _offsetInFile;
  int _byteIncrement;
  Components _component;
  QMap<Data, SignalMember *> _optionalMembers;

  double _countPerVolt;
  double _voltPerUnit;
  AmplitudeUnits _amplitudeUnit;

  SignalDB * _db;

  void clearOptionalMembers();

  // Functions to access optional members;
  SIGNAL_MEMBER_DECL(SMArrayDouble, timePick, TimePick)
  SIGNAL_MEMBER_DECL(SMDuplicateRays, duplicateRays, DuplicateRays)
  SIGNAL_MEMBER_DECL(SMString, comments, Comments)
  SIGNAL_MEMBER_DECL(MiniSeedRecords, miniSeedRecords, MiniSeedRecordList)
  SIGNAL_MEMBER_DECL(GuralpRecords, guralpRecords, GuralpRecordList)

  // Current keys used in sort
  static QList<SortKey> _keys;

  bool writeError( const DoubleSignal * sig, QString logMessage ) const;
  bool readError( const char * buf );

  // Functions to load samples from specific file formats into memory
  bool load( double * samples ) const;
  bool loadGeopsySignal( double * samples ) const;
  bool loadSeg2( double * samples ) const;
  bool loadSu( double * samples, QDataStream::ByteOrder bo ) const;
  bool loadSegY( double * samples, QDataStream::ByteOrder bo ) const;
  bool loadPasscalSegY(double * samples, QDataStream::ByteOrder bo) const;
  bool loadSac( double * samples, QDataStream::ByteOrder bo ) const;
  bool loadRD3( double * samples ) const;
  bool loadNiSismo( double * samples ) const;
  bool loadRadan( double * samples ) const;
  bool loadGse2( double * samples ) const;
  bool loadCity2( double * samples ) const { return loadAscii( samples, "Loading City 2 values " );}
  bool loadAscii( double * samples, QString title ) const;
  bool loadSismalp( double * samples ) const;
  bool loadSyscom3Bytes( double * samples ) const;
  bool loadSyscom2Bytes( double * samples ) const;
  bool loadGuralpGcf( double * samples ) const;
  bool loadWav( double * samples ) const;
  bool loadMiniSeed( double * samples ) const;
  bool loadAsciiOneColumn( double * samples ) const;
  static bool loadMultiChannelsWav( Signal * sig0, double * samples0 );
  static bool loadMultiChannelsAscii( Signal * sig0, double * samples0 );
  static SubSignalPool beginMultiChannelLoad( Signal * sig0, double * samples0,
                                                 int & nSig, double **& samplePtr );
  static void endMultiChannelLoad( Signal * sig0, const SubSignalPool& loadedSigs );
  static void miniSeedRecordHandler ( char *record, int reclen, void *f );
};

inline bool Signal::lessThan( const Signal * sig1, const Signal * sig2 )
{
  return *sig1 < *sig2;
}

inline bool Signal::warnReadOnlySamples() const
{
  TRACE;
  if (_isReadOnlySamples) {
    App::stream() << tr("Trying to modify read only signal %1, aborted").arg(name()) << endl;
    return true;
  } else return false;
}

inline QString Signal::debugName() const
{
  TRACE;
  return _name + "_" + componentLetter(_component) + QString::number( _id );
}

inline double Signal::amplitudeAt( double abscissa ) const
{
  TRACE;
  if ( _type == Waveform )
    return DoubleSignal::amplitudeAt( abscissa - _t0 );
  else
    return DoubleSignal::amplitudeAt( abscissa );
}

inline bool Signal::taper( const TimeRange& tw, double width )
{
  TRACE;
  return DoubleSignal::taper( tw.start() - _t0, tw.end() - _t0, width );
}

inline void Signal::copySamplesFrom( const Signal * sig, const TimeRange& tw )
{
  TRACE;
  DoubleSignal::copySamplesFrom( sig, tw.start() - sig->t0(), tw.start() - t0(), tw.lengthSeconds() );
}

inline int Signal::add( const Signal * sig, const TimeRange& tw, bool checkInvalid, double invalidValue )
{
  TRACE;
  return DoubleSignal::add( sig, tw.start() - sig->t0(), tw.start() - t0(), tw.lengthSeconds(),
                               checkInvalid, invalidValue );
}

inline double Signal::correlation( const Signal * sig, const TimeRange& tw )
{
  TRACE;
  return DoubleSignal::correlation( sig, tw.start() - sig->t0(), tw.start() - t0(), tw.lengthSeconds() );
}

} // namespace GeopsyCore

#endif // SIGNAL_H
