/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-11-10
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/
#ifndef SIGNALFILE_H
#define SIGNALFILE_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"
#include "SignalFileFormat.h"

namespace GeopsyCore {

class Signal;
class SignalDB;
class SubSignalPool;
class GeopsySignalHeader;
class SignalFileFormat;

class GEOPSYCORE_EXPORT SignalFile : public XMLClass
{
  TRANSLATIONS("SignalFile")
public:
  SignalFile(SignalDB * db, QString fileName = "", const SignalFileFormat& format=SignalFileFormat::Unknown);
  SignalFile(SignalDB * db, SignalFile& f, QDir& d );
  ~SignalFile();

  virtual const QString& xml_tagName() const {return xmlSignalFileTag;}
  static const QString xmlSignalFileTag;

  SignalFile& operator=(const SignalFile& p);
  QString name( int len ) const;
  QString shortName() const;
  // Interface function to data members
  QString name() const { return _name;}
  void setName( QString n ) {_name = n;}

  const SignalFileFormat& format() const {return _format;}
  void setFormat(const SignalFileFormat& f) {_format=f;}
  void setFormat(QString f) {_format=SignalFileFormat::fromString(f);}

  bool isOriginalFile() {return _isOriginalFile;}
  void setOriginalFile(bool b) {_isOriginalFile = b;}

  SignalDB * database() const { return _db; }
  void setDatabase(SignalDB * db) { _db = db; }

  // Load signals of the file
  void removeFile();
  bool load();
  bool loadGeopsySignal(const SubSignalPool& subPool);

  static bool setTimeReference( double& t0, QDateTime& timeRef, QDateTime timeTrue );
  static void setSignalName( Signal * sig, QString stationName, QString sufix,
                             int recNum, QString fileName );
  bool save( const QDir& d );
protected:
  virtual void xml_writeProperties( XML_WRITEPROPERTIES_ARGS ) const;
  virtual void xml_writeChildren( XML_WRITECHILDREN_ARGS ) const;
  virtual bool xml_setProperty( XML_SETPROPERTY_ARGS );
  virtual XMLMember xml_member( XML_MEMBER_ARGS );
  virtual void xml_polish(XML_POLISH_ARGS);
private:
  SignalFileFormat _format;
  QString _name;
  bool _isOriginalFile;
  SignalDB * _db;

  bool loadGeopsySignal();
  bool loadCity2();
  bool loadSeg2();
  bool loadSu(QDataStream::ByteOrder bo);
  bool loadSegY(QDataStream::ByteOrder bo);
  bool loadPasscalSegY(QDataStream::ByteOrder bo);
  bool loadRD3();
  bool loadTomo();
  bool loadNiSismo();
  bool loadSac(QDataStream::ByteOrder bo);
  bool loadRadan();
  bool loadGse2(SignalFileFormat::Storage sto=SignalFileFormat::Multi);
  bool loadAsciiOneColumn();
  bool loadAscii();
  bool loadSaf();
  bool loadSismalp();
  bool loadWav();
  bool loadSyscomXmr();
  bool loadSyscomSmr();
  bool loadSyscomVmrx();
  bool loadGuralpGcf();
  bool loadAsciiGeoSig();
  bool loadMiniSeed();

  void warnAsciiHeader();
};

} // namespace GeopsyCore

#endif // SIGNALFILE_H
