/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2006-08-24
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SIGNALMEMBER_H
#define SIGNALMEMBER_H

#include <QGpCoreTools.h>
#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class SubSignalPool;

class GEOPSYCORE_EXPORT SignalMember: public XMLClass
{
public:
  SignalMember() {}
  virtual ~SignalMember() {}

  virtual int count() const {return 1;}
  virtual bool hasIndex(int index) const { return index==0; }

  virtual bool isStored() {return true;}
  virtual bool storeAsXML() {return false;}
  virtual QString toString(int ) const {return QString::null;}
  virtual void fromString(int , QString ) {}

  virtual SignalMember * newCopy() = 0;
};

template <class valueType> class GEOPSYCORE_EXPORT SMNumber: public SignalMember
{
public:
  SMNumber() { _value=0; }
  SMNumber( valueType val ) { _value=val; }
  SMNumber( const SMNumber& o) : SignalMember(o) { _value = o._value; }

  valueType value() const {return _value;}
  void setValue( valueType v) {_value=v;}

  virtual QString toString(int ) const {return QString::number(_value, 'g', 20);}
  virtual SignalMember * newCopy() {return new SMNumber<valueType>(_value);}
private:
  valueType _value;
};

class GEOPSYCORE_EXPORT SMInt: public SMNumber<int>
{
public:
  SMInt() : SMNumber<int>() {}
  SMInt( const SMInt& o) : SMNumber<int>(o) {}

  virtual void fromString(int , QString val) {setValue(val.toInt());}
};

class GEOPSYCORE_EXPORT SMDouble: public SMNumber<double>
{
public:
  SMDouble() : SMNumber<double>() {}
  SMDouble( const SMDouble& o) : SMNumber<double>(o) {}

  virtual void fromString(int , QString val) {setValue(val.toDouble());}
};

template <class valueType> class GEOPSYCORE_EXPORT SMArray: public SignalMember
{
public:
  SMArray() {_count=0;_values=0;}
  SMArray(const SMArray& o);
  ~SMArray() {delete [] _values;}

  valueType value(int index) const { return index<_count ? _values[index] : 0; }
  void setValue( int index, valueType v) {if (index>=_count) enlarge(index); _values[index]=v;}
  virtual int count() const {return _count;}
  virtual bool hasIndex(int index) const { return index<_count && _values[index]!=0; }

  virtual QString toString(int index) const {return QString::number(value(index), 'g', 20);}
  virtual SignalMember * newCopy() {return new SMArray<valueType>(*this);}
private:
  int _count;
  valueType * _values;

  void enlarge( int index );
};

template <class valueType> 
void SMArray<valueType>::enlarge( int index )
{
  TRACE;
  int newSize = _count>0 ? _count : 1;
  while (newSize<=index) newSize = newSize << 1;
  valueType * tmp = new valueType[newSize];
  if (_values) {
    memcpy( tmp, _values, sizeof(valueType)*_count );
    delete [] _values;
  }
  memset( tmp+_count, 0, sizeof(valueType)*(newSize-_count) );
  _values = tmp;
  _count = newSize;
}

template <class valueType> 
SMArray<valueType>::SMArray( const SMArray& o ) : SignalMember(o)
{
  _count=o._count;
  if (o._values) {
    _values = new valueType[_count];
    memcpy( _values, o._values, sizeof(valueType)*_count );
  } else {
    _values = 0;
  }
}

class GEOPSYCORE_EXPORT SMArrayInt: public SMArray<int>
{
public:
  SMArrayInt() : SMArray<int>() {}
  SMArrayInt( const SMArrayInt& o) : SMArray<int>(o) {}

  virtual void fromString(int index, QString val) {setValue( index, val.toInt());}
};

class GEOPSYCORE_EXPORT SMArrayDouble: public SMArray<double>
{
public:
  SMArrayDouble() : SMArray<double>() {}
  SMArrayDouble( const SMArrayDouble& o) : SMArray<double>(o) {}

  virtual void fromString(int index, QString val) {setValue(index, val.toDouble());}
};

class GEOPSYCORE_EXPORT SMDuplicateRays: public SignalMember
{
public:
  SMDuplicateRays() {_rays=0;_id=0;}
  SMDuplicateRays( const SMDuplicateRays& o) : SignalMember(o) {_rays=o._rays;_id=o._id;}

  void setId( int id ) {_id=id;}
  void setRays( SubSignalPool * r ) {_rays=r;}

  int id() const {return _id;}
  SubSignalPool * rays() const {return _rays;}

  virtual bool isStored() {return false;}
  virtual SignalMember * newCopy() {return new SMDuplicateRays(*this);}
private:
  SubSignalPool * _rays;
  int _id;
};


template <class valueType> class GEOPSYCORE_EXPORT SMClass: public SignalMember
{
public:
  SMClass() {}
  SMClass( const SMClass& o) : SignalMember(o) { _value = o._value; }

  valueType value() const {return _value;}
  void setValue( valueType v) {_value=v;}

  virtual SignalMember * newCopy() {return new SMClass<valueType>(*this);}
private:
  valueType _value;
};

class GEOPSYCORE_EXPORT SMString: public SMClass<QString>
{
public:
  SMString() {}
  SMString( const SMClass<QString>& o) : SMClass<QString>(o) {}

  virtual QString toString(int ) const {return value();}
  virtual void fromString(int , QString val) {setValue(val);}
};

} // namespace GeopsyCore

#endif // SIGNALMEMBER_H
