/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-03-26
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SIGNALPROCESS_H
#define SIGNALPROCESS_H

#include "GeopsyCoreDLLExport.h"
#include "RotateParameters.h"
#include "MorletParameters.h"
#include "SubSignalPool.h"

namespace GeopsyCore {

class GEOPSYCORE_EXPORT SignalProcess : public QObject
{
  Q_OBJECT
public:
  SignalProcess( QObject * parent = 0 ) : QObject( parent ) {}

  void setCurrentSubPool( SubSignalPool * subPool );

  const QString& history() const { return _history; }
  int stepCount() const { return _steps.count(); }
  QStringList steps() const;
  void clear();

  int indexOfStep( const QString& tag ) const;

  void saveStep( const QString& tag );
  void restoreStep( int index );

  void fastFourierTransform( DoubleSignal::SignalType st );
  bool exportFile( const QString& filePath, bool useOriginalBaseName = false,
                   const SignalFileFormat& format=SignalFileFormat::Unknown, int pickNumber = 0 );
  void setHeader( int signalIndex, const Signal::DataIndex& dataIndex, const QVariant& value );

  void subtractValue( double val );
  void subtractSignal( int index );
  void subtractSignals( const QString& groupName );
  void multiply( double val );
  void filter( const FilterParameters& param );
  void whiten();
  void agc( double width );
  void stddevClip( double factor );
  void shift( double dt );
  void overSample( double factor );
  void taper( RelativeTimeRange& param, double width );
  void cut( RelativeTimeRange& param );
  bool merge();
  bool mergeStations();
  void decimateAmplitude( int maxCount, double maxRef );
  void decimateTime( int factor );
  void waveletTransform( const MorletParameters& param );
  void stalta( double tsta, double tlta );
  void discreteFourierTransform();
  void rotateComponents( const RotateParameters& param );
  void correlations( double maxDelay, int referenceIndex = -1 );
protected:
  void init();

  void saveStepCore( const QString& tag );
  void restoreStepCore( int index );

  void fastFourierTransformCore( DoubleSignal::SignalType st );
  bool exportFileCore(const QString& filePath, bool useOriginalBaseName,
                      const SignalFileFormat& format, int pickNumber );
  QVariant headerCore( int signalIndex, const Signal::DataIndex& dataIndex );
  void setHeaderCore( int signalIndex, const Signal::DataIndex& dataIndex, const QVariant& value );
  int signalCountCore();

  void subtractValueCore( double val );
  void subtractSignalCore( int index );
  void subtractSignalsCore( const QString& groupName );
  void multiplyCore( double val );
  void filterCore( const FilterParameters& param );
  void whitenCore();
  void agcCore( double width );
  void stddevClipCore( double factor );
  void shiftCore( double dt );
  void overSampleCore( double factor );
  void taperCore( RelativeTimeRange& param, double width );
  void cutCore( RelativeTimeRange& param );
  bool mergeCore();
  bool mergeStationsCore();
  void decimateAmplitudeCore( int maxCount, double maxRef );
  void decimateTimeCore( int factor );
  void waveletTransformCore( const MorletParameters& param );
  void staltaCore( double tsta, double tlta );
  void discreteFourierTransformCore();
  void rotateComponentsCore( const RotateParameters& param );
  void correlationsCore( double maxDelay, int referenceIndex );
protected:
  void addToHistory( const QString& s ) { _history += s; }
private:
  void setPicks( RelativeTimeRange& param );
  static QString timeRangeToArguments( const RelativeTimeRange& param );

  SubSignalPool * _current;
  QVector<SubSignalPool> _steps;
  QString _history;
};

} // namespace GeopsyCore

#endif // SIGNALPROCESS_H
