/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2009-06-01
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SPARSETIMERANGE_H
#define SPARSETIMERANGE_H

#include <QGpCoreTools.h>
#include "TimeRange.h"
#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class GEOPSYCORE_EXPORT SparseTimeRange : private QVector<TimeRange>
{
  TRANSLATIONS("SparseTimeRange")
public:
  SparseTimeRange();
  SparseTimeRange( const SparseTimeRange& r );
  SparseTimeRange( const TimeRange& r );
  ~SparseTimeRange();

  void add(const TimeRange& r, bool acceptOverlap=false);
  void add(const SparseTimeRange& r, bool acceptOverlap=false);
  void remove(const SparseTimeRange& r);
  void remove(const TimeRange& r);
  void clear();
  void shift(double dt);
  void scale(double center, double factor);
  void removeGaps(double dt);
  void removeBlocks(double dt);

  SparseTimeRange invert(bool infiniteEnds=false) const;
  SparseTimeRange invert( const TimeRange& r ) const;
  inline SparseTimeRange intersection( const TimeRange& r ) const;
  SparseTimeRange intersection( const SparseTimeRange& r ) const;
  SparseTimeRange hit(const TimeRange& r) const;
  SparseTimeRange hit(const SparseTimeRange& r) const;
  inline bool intersects(const SparseTimeRange& r) const;
  inline bool intersects(const TimeRange& r) const;

  bool isNull() const {return QVector<TimeRange>::isEmpty();}
  const QVector<TimeRange>& ranges() const { return *this; }
  inline TimeRange range() const;

  void testIndex() const;
  void printDebug() const;
protected:
  int index( double t ) const;
  SparseTimeRange intersection( int index, const TimeRange& r ) const;
  int remove( int& index, const TimeRange& r );
private:
  void append( const TimeRange& r );
  void insert( int before, const TimeRange& r );
  void remove( int index );

  int _n2;
  static const double _relPrec;
};

inline bool SparseTimeRange::intersects( const TimeRange& r ) const
{
  SparseTimeRange res = intersection( r );
  return !res.isEmpty();
}

inline bool SparseTimeRange::intersects(const SparseTimeRange& r) const
{
  SparseTimeRange res=intersection(r);
  return !res.isEmpty();
}

inline SparseTimeRange SparseTimeRange::intersection( const TimeRange& r ) const
{
  if(isEmpty()) return *this;
  return intersection(index(r.start()), r);
}

inline void SparseTimeRange::remove(const TimeRange& r)
{
  if(!isEmpty()) {
    int i=index(r.start());
    remove(i, r);
  }
}

inline TimeRange SparseTimeRange::range() const
{
  if(isEmpty())
    return TimeRange();
  else
    return TimeRange(first().start(), last().end());
}

} // namespace GeopsyCore

#endif // SPARSETIMERANGE_H
