/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-12-24
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef STATIONSIGNALS_H
#define STATIONSIGNALS_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"
#include "SubSignalPool.h"

namespace GeopsyCore {

class SparseKeepSignal;

class GEOPSYCORE_EXPORT StationSignals
{
  TRANSLATIONS("StationSignals")
public:
  enum Component {UndefinedComponent = 0, VerticalComponent=1, HorizontalComponent=2, AllComponent=3,
                  AnySingleComponent=7, AnyComponent=11};
  Q_DECLARE_FLAGS(Components, Component)

  StationSignals( Components c = UndefinedComponent );
  virtual ~StationSignals();

  void setComponents( Components c );

  Components components() const { return _components; }
  int nComponents() const { return nComponents( _components ); }
  static int nComponents( Components c );
  static Signal::Components component( int index, Components c );

  bool addSignal( Signal * sig, QString * log = 0 );
  bool hasAllComponents( QString * log = 0 ) const;
  bool contains( Signal * sig ) const;
  static void organizeSubPool( SubSignalPool * subPool );

  const SubSignalPool& originals(int iComp) const {return _signals[iComp];}

  double minTime() const;
  double maxTime() const;

  // General information on station deduced from first valid station
  Signal * firstValidSignal() const;
  Signal * lastValidSignal() const;
  inline double deltaT() const;
  inline double t0() const;
  inline QDateTime timeReference() const;
  inline QString name() const;
  inline QString nameComponent() const;
  inline Signal::Components component() const;
  inline const Point& coordinates() const;
  inline const Point& source() const;
  inline double sourceReceiverDistance() const;
  inline double sourceReceiverAzimuth() const;

  SparseTimeRange timeRange(const TimeRange& r) const;
  void setSampling(SparseKeepSignal * keep) const;
  void setKeep(SparseKeepSignal * keep, const WindowingParameters& param, int stationIndex) const;
protected:
  SubSignalPool * _signals;
private:
  Components _components;
};

inline double StationSignals::deltaT() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->deltaT() : 0.0;
}

inline double StationSignals::t0() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->t0() : 0.0;
}

inline QDateTime StationSignals::timeReference() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->timeReference() : QDateTime();
}

inline QString StationSignals::name() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->name() : tr("Unamed");
}

inline QString StationSignals::nameComponent() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->nameComponent() : tr("Unamed");
}

inline Signal::Components StationSignals::component() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->component() : Signal::UndefinedComponent;
}

inline const Point& StationSignals::coordinates() const
{
  static Point dummy;
  Signal * sig = firstValidSignal();
  return sig ? sig->receiver() : dummy;
}

inline const Point& StationSignals::source() const
{
  static Point dummy;
  Signal * sig = firstValidSignal();
  return sig ? sig->source() : dummy;
}

inline double StationSignals::sourceReceiverDistance() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->sourceReceiverDistance() : 0.0;
}

inline double StationSignals::sourceReceiverAzimuth() const
{
  TRACE;
  Signal * sig = firstValidSignal();
  return sig ? sig->sourceReceiverAzimuth() : 0.0;
}

} // namespace GeopsyCore

Q_DECLARE_OPERATORS_FOR_FLAGS(GeopsyCore::StationSignals::Components)

#endif // STATIONSIGNALS_H
