/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-11-10
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SUBSIGNALPOOL_H
#define SUBSIGNALPOOL_H

#include <QGpCoreTools.h>
#include "GeopsyCoreDLLExport.h"
#include "Signal.h"
#include "RelativeTimeRange.h"

namespace GeopsyCore {

class SignalPool;
class SignalDataPool;
class SignalGroup;

extern const char * SignalKeyName [];

class GEOPSYCORE_EXPORT SubSignalPool : protected QList<Signal *>
{
  TRANSLATIONS( "SubPoolWindow" )
public:
  SubSignalPool() {}
  SubSignalPool( const SubSignalPool& subPool );
  ~SubSignalPool() { removeAll(); }

  // Useful members from QList with public access
  typedef QList<Signal *>::iterator iterator;
  typedef QList<Signal *>::const_iterator const_iterator;
  iterator begin() { return QList<Signal *>::begin();}
  iterator end() { return QList<Signal *>::end();}
  const_iterator begin() const { return QList<Signal *>::begin();}
  const_iterator end() const { return QList<Signal *>::end();}
  SubSignalPool& operator=( const SubSignalPool& o );
  bool operator==( const SubSignalPool& o ) const;
  bool operator!=( const SubSignalPool& o ) const { return ! operator==( o ); }
  SubSignalPool copy() const;
  int count() const { return QList<Signal *>::count(); }
  int samplesCount();
  bool empty() { return QList<Signal *>::empty();}
  bool isEmpty() const { return QList<Signal *>::isEmpty();}
  Signal * at( int index ) const { return QList<Signal *>::at( index );}
  Signal * first() const { return QList<Signal *>::first();}
  Signal * last() const { return QList<Signal *>::last();}
  int indexOf( const Signal * sig ) const { return QList<Signal *>::indexOf( const_cast<Signal *>(sig) );}
  bool contains( const Signal * sig ) const { return QList<Signal *>::contains( const_cast<Signal *>(sig) );}

  void sort() { qSort( begin(), end(), Signal::lessThan ); }

  const QString& name() const { return _name; }
  void setName( const QString& n ) { _name = n; }

  void setHeaderModified( bool b ) const;
  bool isHeaderModified() const;
  bool isReadOnlySamples() const;

  TimeRange timeRange() const;

  // Signal selection
  void addAll();
  void addSubPool( const SubSignalPool& subPool );
  void addSource( Point point );
  void addReceiver( Point point );
  void addSignal( Signal * sig ) { append( sig ); }
  void addSignal( int id );
  void addFile( int fileN );
  void addFile( const SignalFile * sf );
  void addGroup( SignalDB * db, SignalGroup * g );
  void addList( QString signalList );
  void insertSignal( Signal * sig, Signal * refSig = 0, bool before = true );
  void remove( Signal * sig );
  void removeAt( int i );
  void remove( Signal::Components comp );
  void removeAll();
  void debugSignalSharing();

  // Saving to file
  static bool isExportFormat(const SignalFileFormat& f);
  bool save(QString filePath, bool useOriginalBaseName, SignalFileFormat format,
            int maximumSignalsPerFile=0, int pickNumber=0 ) const;
  bool saveGeopsySignal ( QString filePath ) const;

  // Signal processing
  void pickCoppens( int pickno, double delta, double tend, double tstart = 0., bool relative = false, double mute = 0. );
  void pickMinMax( int pickno, double from, double to, Signal::PickWhat what );
  void subtractValue( double val = 0.0 );
  void subtractSignal( Signal * bysig );
  void subtractSignals( SubSignalPool * bySubPool );
  void filter( const FilterParameters& param );
  void waveletTransform( const MorletParameters& param );
  void shift( double dt );
  void overSample( double factor );
  void whiten();
  void agc( double width );
  void stddevClip( double factor );
  void multiply( double value );
  bool mergeStations();
  bool merge();
  void stalta( double tsta, double tlta );
  void taper( const RelativeTimeRange& param, double width );
  void cut( const RelativeTimeRange& param );
  void correlations( double maxDelay, const Signal * referenceSig = 0 );
  void decimateAmplitude( int maxCount, double maxRef );
  void decimateTime( int factor );
  void fastFourierTransform( DoubleSignal::SignalType st );
  void discreteFourierTransform();
  void importTable( QString fname );
  void exportTable( QString fname, SignalDataPool& param );
  int linearSearch( const Point& source, const Point& receiver );
  void unglitch( double threshold );
  int checkDuplicateRays();
  void rotateComponents( Matrix3x3 rotMatrix );
  bool associate3Components();
  Curve<Point> sources() const;
  Curve<Point> receivers() const;
  double azimuth() const;
  int maximumTimePickCount() const;
protected:
  QString _name;
private:
  bool exportFilePath(QString& filePath, bool useOriginalBaseName, const SignalFileFormat& f) const;
  bool check3ComponentTriplet( SubSignalPool& triplet );
  inline void prepend( Signal * sig );
  inline void append( Signal * sig );

  bool saveSeg2( QString filePath ) const;
  bool saveSeg2Return( bool ret, int * offset ) const;
  bool writeSeg2Header( QFile& f, int nSignals ) const;
  bool writeSeg2Offsets( QFile& f, int nSignals, int *offset ) const;
  bool saveArrivalTimes ( QString filePath, int pickNo ) const;
  bool saveAsciiOneColumn( QString filePath ) const;
  bool saveAsciiMultiColumns( QString filePath ) const;
  bool saveSaf( QString filePath ) const;
  bool saveOneSaf( QString filePath ) const;
  bool saveSac( QString filePath, bool useOriginalBaseName, QDataStream::ByteOrder bo ) const;
  bool saveSu( QString filePath, QDataStream::ByteOrder bo ) const;
  bool saveWav( QString filePath ) const;
  bool saveGse2( QString filePath ) const;
  bool saveMiniSeed( QString filePath ) const;
};

inline void SubSignalPool::append( Signal * sig )
{
  TRACE;
  if (sig) {
    sig->addReference();
    QList<Signal *>::append( sig );
  }
}

inline void SubSignalPool::prepend( Signal * sig )
{
  TRACE;
  if (sig) {
    sig->addReference();
    QList<Signal *>::prepend( sig );
  }
}

} // namespace GeopsyCore

#endif // SUBSIGNALPOOL_H
