/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-09-17
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef TIMERANGE_H
#define TIMERANGE_H

#include <QtCore>
#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class GEOPSYCORE_EXPORT TimeRange
{
public:
  inline TimeRange();
  inline TimeRange( const TimeRange& o ) { operator=(o); }
  inline TimeRange( double start, double end);
  ~TimeRange() {}

  inline void operator=( const TimeRange& o );

  double start() const {return _start;}
  double end() const {return _end;}

  inline void setStart( double m );
  inline void setEnd( double m );

  inline void shift( double s );
  inline void scale( double center, double factor );

  int lengthSamples(double deltaT) const { return (int) floor(lengthSeconds()/deltaT + 0.5 ); }
  double lengthSeconds() const {return _end-_start;}

  inline TimeRange intersection( const TimeRange& tw ) const;
  inline bool intersects( const TimeRange& tw ) const;

  void printDebug() const;
private:
  double _start;
  double _end;
};

inline TimeRange::TimeRange()
{
  _start = 0.0;
  _end = 0.0;
}

inline TimeRange::TimeRange( double start, double end )
{
  if (start>end) {
    _start = end;
    _end = start;
  } else {
    _start = start;
    _end = end;
  }
}

inline void TimeRange::operator=( const TimeRange& o )
{
  _start = o._start;
  _end = o._end;
}

/*!
  Set start time. Swap end and start if not in correct order.
*/
inline void TimeRange::setStart(double m)
{
  _start=m;
}

/*!
  Set end time. Swap end and start if not in correct order.
*/
inline void TimeRange::setEnd(double m)
{
  _end=m;
}

inline void TimeRange::shift( double s )
{
  _start+=s;
  _end+=s;
}

inline void TimeRange::scale( double center, double factor )
{
  _start=center+(_start-center)*factor;
  _end=center+(_end-center)*factor;
}

inline TimeRange TimeRange::intersection( const TimeRange& tw ) const
{
  TimeRange res;
  if ( tw._end < _start || tw._start > _end )
    return res;
  if ( tw._start < _start && _start < tw._end )
    res._start = _start;
  else
    res._start = tw._start;
  if ( tw._start < _end && _end < tw._end )
    res._end = _end;
  else
    res._end = tw._end;
  return res;
}

inline bool TimeRange::intersects( const TimeRange& tw ) const
{
  if ( tw._end < _start || tw._start > _end )
    return false;
  return true;
}

} // namespace GeopsyCore

// Allow passing TimeRange as a QVariant or through SIGNAL/SLOT
using namespace GeopsyCore;
Q_DECLARE_METATYPE(TimeRange);

#endif // TIMERANGE_H
