/***************************************************************************
**
**  This file is part of MonoStation.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2007-08-17
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTSTATION_H
#define ABSTRACTSTATION_H

#include <GeopsyCore.h>
#include <GeopsyGui.h>

#include "StationResults.h"
#include "MonoStationDLLExport.h"

namespace MonoStation {

class SpectrumParameters;
class WinResults;

class MONOSTATION_EXPORT AbstractStation : public QObject
{
  Q_OBJECT
public:
  AbstractStation( StationProcessSignals * sig );
  virtual ~AbstractStation();

  void setParent(WinResults * p) {_parent=p;}
  WinResults * parent() const {return _parent;}
  int indexOf(StationResults * c) {return _results.indexOf(c);}

  virtual void start(SpectrumParameters& param) = 0;
  virtual void start(const TimeRangeList& twList, SpectrumParameters& param) = 0;
  void save( QDir outputDir );
  virtual void save( int index, QDir outputDir ) = 0;

  virtual int graphCount() const { return _results.count(); }
  virtual void setResultGraph( int index, AxisWindow * w ) { _results[index]->setGraph(w); }
  virtual AxisWindow * resultGraph(int index) const {return _results[index]->graph();}

  virtual void setResultLabel( int index, TextEdit * w ) { _results[index]->setLabel(w); }
  virtual TextEdit * resultLabel( int index ) const { return _results[index]->label(); }

  void clearResults();
  virtual const StationResults * results(int index) const { return index<graphCount() ? _results[index] : 0; }
  virtual StationResults * results(int index) { return index<graphCount() ? _results[index] : 0; }
  virtual double resultMaxAmplitude() const;
  void setHighPassFilter( double frequency ) { _signals->setHighPassFilter(frequency); }
  virtual bool isSelected() const;

  virtual QString name( int index ) const = 0;
  virtual QString name() const = 0;
  QString shortFileName() const;
  const StationSignals * originalSignals() const { return _signals->originalSignals(); }
  int nComponents() const { return _signals->nComponents(); }

  SparseKeepSignal * keep(const TimeRange& r, const WindowingParameters& param, int stationIndex) const;
  SparseTimeRange timeRange(const TimeRange& r) const {return _signals->originalSignals()->timeRange(r);}
  void setSampling(SparseKeepSignal * keep) const {_signals->originalSignals()->setSampling(keep);}
  void setKeep(SparseKeepSignal * keep, const WindowingParameters& param, int stationIndex) const {_signals->originalSignals()->setKeep(keep,param,stationIndex);}

  void addWindows(const TimeRange& r, const WindowingParameters& param, const SparseKeepSignal& keep);
  void addWindows(QTextStream& s) {_timeWindows.add(s, &_winLog); _windowsChanged = true;}
  void removeWindows(const TimeRange& r);
  void clearAllWindows();
  void clearGrayWindows();
  void setWindowColors();
  bool hasGrayWindows();
  void inverseWindows(const TimeRange& r, const WindowingParameters& param, const SparseKeepSignal& keep);
  bool windowsModified() { return _windowsChanged; }
  TimeRange windowRange() const {return _timeWindows.timeRange();}
  int windowCount() const {return _timeWindows.count();}
  TimeWindowList& windows() {return _timeWindows;}
  const TimeWindow& windowAt(int index) const {return _timeWindows.at(index);}

  void setLog(QString& log) {_log = log;}
  QString windowingLog() const {return _winLog;}
  void clearWindowingLog() {_winLog.clear();}
  void addWindowingLog(const QString& t) {_winLog+=t;}
public slots:
  void changeColors(const QVector<int>* selCurves, const QColor& col);
signals:
  void windowsChanged();
protected:
  SparseKeepSignal * _keep;
  StationProcessSignals * _signals;
  TimeWindowList _timeWindows;

  bool _windowsChanged;
  QString _log, _winLog;

  WinResults * _parent;
  QVector<StationResults *> _results;
};

} // namespace MonoStation

#endif // ABSTRACTSTATION_H
