/***************************************************************************
**
**  This file is part of MonoStation.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2007-08-17
**  Authors:
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef STATIONRESULTS_H
#define STATIONRESULTS_H

#include <GeopsyCore.h>
#include <SciFigs.h>

#include "MonoStationDLLExport.h"

namespace MonoStation {

class AbstractStation;

class MONOSTATION_EXPORT StationResults : public QObject
{
  Q_OBJECT
public:
  StationResults();

  void setParent(AbstractStation * p) {_parent=p;}
  AbstractStation * parent() const {return _parent;}

  void setGraph(AxisWindow * w);
  AxisWindow * graph() const { return _averageLayer->graph(); }

  void setLabel(TextEdit * l) { _label = l; }
  TextEdit * label() const { return _label; }

  void clear();

  void setWindows( int winCount, Sampling& samp );
  void addWindows( int winCount );
  int windowCount() const { return _winCount; }
  void setWindow( int winIndex, DoubleSignal * sig );
  void setWindowColor( int winIndex, const QColor& col );
  void setWindowColor( const QVector<int>* selCurves, const QColor& col );
  void setT10Layer( double averageWindowLength );
  void setAverageLayer();
  void setPeakLayer();
  void setVisibleLayers(int visibleLayers);
  void save(QString fileName, QString log);
  void load( QString fileName ) { loadAverage( fileName ); }

  const QString& category() const {return _category;}
  void setCategory(const QString& c);

  const QString& stationName() const {return _stationName;}
  void setStationName(const QString& n) {_stationName=n;}

  const Point& coordinates() const {return _coordinates;}
  void setCoordinates(const Point& p) {_coordinates=p;}

  const Curve<Point2D>& averageCurve() const;
  const Curve<Point2D>& minimumCurve() const;
  const Curve<Point2D>& maximumCurve() const;

  void clearPeaks();
  void addPeak(double minFreq, double maxFreq);
  void setPeak(int index, double minFreq, double maxFreq);
  void removePeak(int index);

  int peakCount() const;
  int windowPeakCount(int ib) const {return _nWindowsForPeaks.at(ib);}
  double peak(int ib) const;
  double peakMinimum(int ib) const;
  double peakMaximum(int ib) const;
  double lowestPeak() const;
  double highestPeak() const;
  double maximumAmplitudePeak() const;
  double maximumAmplitude() const;

  void printAverage() const;
  void printWindows() const;
  void printPeaks() const;

  void derivate();
  void integrate();

  ParallelBands * t10Layer() const {return _t10Layer;}
  ParallelBands * peakLayer() const {return _peakLayer;}
  XUniqueYColorLines * windowsLayer() const {return _windowsLayer;}
  LineLayer * averageLayer() const {return _averageLayer;}
  LineLayer * stddevLayer() const {return _stddevLayer;}
private slots:
  void blackWindows( int, const QVector<int>* selCurves );
signals:
  void colorsToChange( const QVector<int>* selCurves, const QColor& col );
  void showColors( bool );
  void peakChanged();
protected:
  virtual QString comments() const;
private:
  Curve<Point2D>& averageCurve();
  Curve<Point2D>& stddevCurve(int index);
  void createAverageLines();
  void setPeak(int index, int minFreq, int maxFreq);
  int averageMax(const Curve<Point2D>& meanCurve) const;
  int max(const double * y, int minFreq, int maxFreq) const;
  int frequency2Index(const double * x, double f);
  void saveLog(QString fileName, QString log);
  void saveAverage(QString fileName);
  void loadAverage(QString fileName);
  void setToolTips();

  AbstractStation * _parent;

  ParallelBands * _t10Layer;
  ParallelBands * _peakLayer;
  XUniqueYColorLines * _windowsLayer;
  LineLayer * _averageLayer;
  LineLayer * _stddevLayer;
  TextEdit * _label;

  QString _category;
  QString _stationName;
  Point _coordinates;
  int _winCount;
  int _nSamples;
  int _nSamples2;
  QVector<int> _nWindowsForPeaks;
};

} // namespace MonoStation

#endif // STATIONRESULTS_H
