/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-09-11
**  Authors:
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ABSTRACTSTREAM_H
#define ABSTRACTSTREAM_H

#include <QtCore>

#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

  class QGPCORETOOLS_EXPORT AbstractStream
  {
  public:
    AbstractStream();
    virtual ~AbstractStream();

    // rtti not available under Windows, let identification of stream type
    enum ClassId {StandardId=0, StringId, FileId, ForwardId, LogViewId, CustomId};
    virtual ClassId classId() const=0;

    void setPrefix(const QString & prefix);

    void freeze(bool b) { _frozen = b;}
    bool frozen() const {return _frozen;}

    void setRemovable(bool r) {_removable=r;}
    bool removable() {return _removable;}

    AbstractStream& operator<<(const QString& val);
    AbstractStream& operator<<(int val) {return operator<<(QString::number(val));}
    AbstractStream& operator<<(double val) {return operator<<(QString::number(val));}
    AbstractStream& operator<<(const char * val) {return operator<<(QString(val));}
    AbstractStream& operator<<(QChar val) {return operator<<(QString(val));}
    AbstractStream& operator<<(char val) {return operator<<(QString(QChar::fromAscii(val)));}
    void endl();
    void flush();
  protected:
    virtual void sendToStream(const QString& val)=0;
    virtual void flushStream()=0;

    void lock() {_mutex.lock();}
    void unlock() {_mutex.unlock();}
  private:
    inline QString timeStampedPrefix();

    bool _frozen;
    bool _removable;
    bool _lineOpen;
    enum PrefixType {NoPrefix, String, TimeStamp};
    PrefixType _prefixType;
    QString _prefix1, _prefix2;
    QMutex _mutex;
  };

  typedef AbstractStream& (*AbstractStreamFunction)(AbstractStream&); // manipulator function

  inline AbstractStream& operator<<(AbstractStream &s, AbstractStreamFunction f)
  {
    return (*f)(s);
  }

  QGPCORETOOLS_EXPORT AbstractStream& endl(AbstractStream &s);
  QGPCORETOOLS_EXPORT AbstractStream& flush(AbstractStream &s);

} // namespace QGpCoreTools

#endif // ABSTRACTSTREAM_H
