/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2006-07-06
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef CACHEITEM_H
#define CACHEITEM_H

#include <time.h>

#include "Global.h"
#include "ApplicationClock.h"
#include "QGpCoreToolsDLLExport.h"
#include "Trace.h"

namespace QGpCoreTools {

  class CacheProcess;

  class QGPCORETOOLS_EXPORT CacheItem
  {
  public:
    inline CacheItem( );
    inline CacheItem( const CacheItem& o );
    inline virtual ~CacheItem();

    inline void addProcess(CacheProcess * p) const;
    inline void removeProcess(CacheProcess * p) const;
    quint64 nextPredictedTime() const;
  protected:
    friend class Cache;
    friend class AllocatedCacheItem;

    inline void lockData() const;
    inline void unlockData() const;

    // Memory management for cached data
    bool isSaved() const { return _saved; }
    virtual bool isAllocated() const = 0;
    virtual int dataSize() const = 0;
    virtual bool allocate() const = 0;
    virtual void free() const = 0;
    virtual void save( QDir& d ) const = 0;
    virtual void load( QDir& d ) const = 0;

    inline QString swapFileName() const { return QString("cache_data_%1").arg( ( ulong ) this, 0, 16 ); }
    virtual QString debugName() const { return swapFileName(); }
  private:
    mutable QMutex _dataLock, _processLock;
    mutable int _lockCount;
    mutable QSet<CacheProcess *> _processes;
    mutable quint64 _lastAccess;
    mutable bool _saved:1;
  };

  inline CacheItem::CacheItem( ) : _dataLock(QMutex::Recursive)
  {
    TRACE;
    _lastAccess=ApplicationClock::elapsed();
    _saved=false;
    _lockCount=0;
  }

  inline CacheItem::CacheItem( const CacheItem& /*o*/ ) : _dataLock(QMutex::Recursive)
  {
    TRACE;
    _lastAccess=ApplicationClock::elapsed();
    _saved=false;
    _lockCount=0;
  }

  inline CacheItem::~CacheItem()
  {
    ASSERT(_lockCount==0);
    ASSERT(!_saved);
  }

  inline void CacheItem::lockData() const
  {
    _dataLock.lock();
    _lockCount++;
  }

  inline void CacheItem::unlockData() const
  {
    _lockCount--;
    _lastAccess=ApplicationClock::elapsed();
    _dataLock.unlock();
  }

  inline void CacheItem::addProcess(CacheProcess * p) const
  {
    _processLock.lock();
    _processes.insert(p);
    _processLock.unlock();
  }

  inline void CacheItem::removeProcess(CacheProcess * p) const
  {
    _processLock.lock();
    _processes.remove(p);
    _processLock.unlock();
  }

} // namespace QGpCoreTools

#endif // CACHEITEM_H
