/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2007-02-08
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef COREAPPLICATIONPRIVATE_H
#define COREAPPLICATIONPRIVATE_H

#include "QGpCoreToolsDLLExport.h"
#include "PackageInfo.h"
#include "AbstractStream.h"
#include "Message.h"
#include "ApplicationHelp.h"

namespace QGpCoreTools {

class PathTranslator;
class ApplicationHelp;

class QGPCORETOOLS_EXPORT GlobalObject
{
public:
  GlobalObject() {}
  virtual ~GlobalObject() {}
};

class QGPCORETOOLS_EXPORT CoreApplicationPrivate
{
  TRANSLATIONS("CoreApplicationPrivate")
protected:
  CoreApplicationPrivate() {int argc=0; init(argc, 0);}
public:
  CoreApplicationPrivate(int & argc, char ** argv) {init(argc, argv);}
  virtual ~CoreApplicationPrivate();

  static QString version();
  static QString version(QString item);
  static QString authors();
  virtual bool hasGui() const=0;

  static void checkBlocks(bool showLib, bool showUndeleted, const char * file=0, int line=0);
  static bool isBlockDamaged(void* v, const char * file, int line);
  static void printBlockInfo(void * ptr, const char * file, int line);
  static void setMemoryVerbose(bool v);

  static bool checkOptionArg(int& i, int argc, char ** argv, bool mandatory=true);
  void showHelp(int& i, int argc, char ** argv, ApplicationHelp * (*help)());

  static CoreApplicationPrivate * instance() {return _self;}

  void debugUserInterrupts(bool d);
  static QString backTrace();
  static QString backTraceBug();
  static QString bugInfo(Message::Severity sev, const QString& msg);
  static QString systemInfo();
  static QString appInfo();
  static QString platform();
  static void osSignal( int sigNum );
  static void messageOutput(QtMsgType type, const char *msg);
  static void messageOutputAbort(QtMsgType type, const char *msg);
  void reportBug(Message::Severity sev, const char * msg);
  void reportBugNow(Message::Severity sev, const char * msg);
  QThread * mainThread() const {return _thread;}
  static void sleep( int ms );
  static QString currentThreadName();

  Message * messageHandler() const {return _messageHandler;}
  QString translatePath(const QString& file, const QString& title, const QString& fileType);

  void clearStreams();
  void initStream(QThread * thread=0);
  void setStream(AbstractStream * stream, QThread * thread=0);
  void freezeStream(bool b) const;
  void streamDestroyed(AbstractStream * stream);
  void setStreamPrefix(const QString& prefix) {currentStream().setPrefix(prefix);}
  inline AbstractStream& currentStream() const;
  AbstractStream& stream(QThread * thread) const;

  static void addGlobalObject(GlobalObject * ptr);
  static void removeGlobalObject(GlobalObject * ptr);

  static QString getStdin();
  static QStringList getFileList(int argc, char ** argv);
  static int userId();
  int terminalRows() const;
  int terminalCols() const;

  void addPlugin(QObject * p);
  void deletePlugin(QObject * p);

  static QString htmlBugReport(const QString& systemInfo, const QString& email=QString::null,
                               const QString &userInfo=QString::null);

  int maximumThreadCount() const {return _maximumThreadCount;}
protected:
  void setTerminalSize();
  void askForUserInterrupt();
  void setMessageHandler( Message * m );
  static QString constructorApplicationName();
  void destructorCleanUp();
  void initInternalDebugger(bool reportBugs);
  void initTranslations();
  void checkAtomicOperations();
  virtual void setHelp( ApplicationHelp * h );
private:
  void terminalSize() const;
  void init ( int & argc, char ** argv );

  static CoreApplicationPrivate * _self;
  static QString _crashingThread;

  QMap< QThread *, AbstractStream *> _streams;
  QThread * _thread;
  Message * _messageHandler;
  PathTranslator * _pathTranslator;
  QStringList _errors;
  QList<GlobalObject *> _globalPointers;
  QList<QObject *> _plugins;
  int _maximumThreadCount;
  bool _reportBugs, _debugUserInterrupts;


  mutable int _terminalCols, _terminalRows;
};

inline AbstractStream& CoreApplicationPrivate::currentStream() const
{
  QMap<QThread *, AbstractStream *>::ConstIterator it = _streams.find(QThread::currentThread());
  if (it!=_streams.end()) return *it.value(); else return *_streams[_thread];
}

/*inline AbstractStream& endl(CoreApplicationPrivate &s)
{
  AbstractStream& ls = s.currentStream();
  ls << ::endl;
  return ls;
}

inline AbstractStream& operator<<(CoreApplicationPrivate &s, QTextStreamFunction f)
{
  AbstractStream& ls=s.currentStream();
  (*f)(ls);
  return ls;
}

inline AbstractStream& operator<<(CoreApplicationPrivate &s, QTextStreamManipulator m)
{
  AbstractStream& ls=s.currentStream();
  m.exec(ls);
  return ls;
}*/

// Shorter name to access application function through static instance
class App
{
public:
  static AbstractStream& stream() {return CoreApplicationPrivate::instance()->currentStream();}
  static void setStream(AbstractStream * stream, QThread * thread = 0) {
    return CoreApplicationPrivate::instance()->setStream(stream, thread);
  }
  static AbstractStream& stream(QThread * thread) {
    return CoreApplicationPrivate::instance()->stream(thread);
  }
  static void freezeStream(bool b) {
    CoreApplicationPrivate::instance()->freezeStream(b);
  }
  static void setStreamPrefix(const QString& prefix) {
    CoreApplicationPrivate::instance()->setStreamPrefix(prefix);
  }
  static void sleep(int ms) {
    CoreApplicationPrivate::sleep(ms);
  }
};

// Convenient macro definitions with code localization
#ifdef MEMORY_CHECKER
#define checkblocks(showLib, showUndeleted) CoreApplicationPrivate::checkBlocks(showLib, showUndeleted, __FILE__, __LINE__)
#define isBlockDamaged(ptr) CoreApplicationPrivate::isBlockDamaged(ptr, __FILE__, __LINE__)
#define printBlockInfo(ptr) CoreApplicationPrivate::printBlockInfo(ptr,__FILE__, __LINE__)
#endif

} // namespace QGpCoreTools

#endif // COREAPPLICATIONPRIVATE_H
