/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-05-17
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MATRIX_H
#define MATRIX_H


#include "MatrixData.h"
#include "Point.h"
#include "Complex.h"

namespace QGpCoreTools {

template <typename T> class MatrixIterator;
template <typename T> class MutableMatrixIterator;
template <typename T> class MatrixRowIterator;
template <typename T> class MutableMatrixRowIterator;
template <typename T> class MatrixColumnIterator;
template <typename T> class MutableMatrixColumnIterator;

template <typename T> class QGPCORETOOLS_EXPORT Matrix
{
public:
  Matrix() { _d = new MatrixData<T>; }
  Matrix( int ndim ) { _d = new MatrixData<T>; _d->resize(ndim); }
  Matrix( int nrow, int ncol ) { _d = new MatrixData<T>; _d->resize(nrow, ncol); }

  Matrix<T> operator=( const Matrix<T>& m ) { *_d = *m._d; return *this; }
  bool operator==( const Matrix<T>& m ) const { return _d->operator==( *m._d ); }
  Matrix<T> operator+( const Matrix<T>& m ) const { return _d->operator+( *m._d ); }
  void operator+=( const Matrix<T>& m );
  Matrix<T> operator*( const Matrix<T>& m ) const { return _d->operator*( *m._d ); }
  void operator*=( const Matrix<T>& m );
  void zero() { _d->zero(); }
  void identity() { _d->identity(); }
  T& at( int row, int col ) { return _d->at(row,col); }
  const T& at( int row, int col ) const { return _d->at(row,col); }
  Matrix<T> transposed() const { return _d->transposed(); }
  void transpose() { *this = transposed(); }

  int columnCount() const { return _d->columnCount(); }
  int rowCount() const { return _d->rowCount(); }
  void resize( int ndim ) { _d->resize(ndim); }
  void resize( int nrow, int ncol ) { _d->resize(nrow, ncol); }

  void print() const {_d->print();}
protected:
  friend class MatrixData<T>;
  friend class MatrixIterator<T>;
  friend class MutableMatrixIterator<T>;
  friend class MatrixRowIterator<T>;
  friend class MutableMatrixRowIterator<T>;
  friend class MatrixColumnIterator<T>;
  friend class MutableMatrixColumnIterator<T>;

  QSharedDataPointer< MatrixData<T> > _d;
};


template <typename T> Matrix<T> MatrixData<T>::operator+( const MatrixData<T>& m ) const
{
  ASSERT(_nrow == m._nrow && _ncol == m._ncol);
  Matrix<T> r( _nrow, _ncol );
  T * rValues = r._d->values();
  for (int i=_nrow * _ncol-1; i>=0;i--) rValues[i] = _values[i] + m._values[i];
  return r;
}

template <typename T> Matrix<T> MatrixData<T>::operator*( const MatrixData<T>& m ) const
{
  ASSERT( _ncol == m._nrow );
  Matrix<T> r( _nrow, m._ncol );
  r.zero();
  if (MatrixMultiply::isValid(_nrow, _ncol, m._ncol)) {
    const MatrixMultiply& multiplier = *MatrixMultiply::begin(_nrow, _ncol, m._ncol);
    T * rValues = r._d->_values;
    const T * thisValues = _values;
    const T * mValues = m._values;
    int n = multiplier.indexCount();
    for(int i=0;i<n;i++) {
      const MatrixMultiply::IndexMap& index = multiplier.indexMap(i);
      rValues[index.resultIndex()]+=thisValues[index.factor1Index()]*mValues[index.factor2Index()];
    }
    MatrixMultiply::end(&multiplier);
  } else {
    ASSERT(false);
    // TODO big matrix multiplication: call to LAPACK?
  }
  return r;
}

template <typename T> Matrix<T> MatrixData<T>::transposed() const
{
  Matrix<T> t(_ncol, _nrow);
  T * pt = t._d->values();
  const T * prow;
  int j;
  for( int i = 0; i < _nrow; i++ ) {
    prow = _values + i * _ncol;
    for ( j=0; j < _ncol; j++ ) {
      (*pt++) = prow[j];
    }
  }
  return t;
}

template <typename T> inline void Matrix<T>::operator*=( const Matrix<T>& m )
{
  *this = _d->operator*( *m._d );
}

template <typename T> inline void Matrix<T>::operator+=( const Matrix<T>& m )
{
  *this = _d->operator+( *m._d );
}

} // namespace QGpCoreTools

#endif // MATRIX_H
