/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2008-07-15
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MATRIXDATA_H
#define MATRIXDATA_H


#include "Global.h"
#include "MatrixMultiply.h"
#include "Number.h"

namespace QGpCoreTools {

template <typename T> class Matrix;

template <typename T> class MatrixData : public QSharedData
{
public:
  inline MatrixData();
  inline MatrixData( const MatrixData<T>& m );
  inline ~MatrixData();

  inline void operator=( const MatrixData<T>& m );
  inline bool operator==( const MatrixData<T>& m ) const;
  Matrix<T> operator+( const MatrixData<T>& m ) const;
  Matrix<T> operator*( const MatrixData<T>& m ) const;
  inline void zero();
  inline void identity();
  inline T& at( int row, int col );
  inline const T& at( int row, int col ) const;
  Matrix<T> transposed() const;

  int columnCount() const { return _ncol; }
  int rowCount() const { return _nrow; }
  T * values() { return _values; }
  const T * values() const { return _values; }
  inline void resize( int ndim ) { resize( ndim, ndim ); }
  inline void resize( int nrow, int ncol );

  void print() const;
protected:
  int _nrow, _ncol;
  T * _values;
};

template <typename T> inline MatrixData<T>::MatrixData()
   : QSharedData()
{
  _nrow = 0;
  _ncol = 0;
  _values = 0;
}

template <typename T> inline MatrixData<T>::MatrixData( const MatrixData<T>& m )
   : QSharedData()
{
  _values = 0;
  resize(m._nrow, m._ncol);
  memcpy( _values, m._values, sizeof(T)*_nrow * _ncol );
}

template <typename T> inline MatrixData<T>::~MatrixData()
{
  delete [] _values;
}

template <typename T> inline void MatrixData<T>::operator=( const MatrixData<T>& m )
{
  resize(m._nrow, m._ncol);
  memcpy( _values, m._values, sizeof(T)*_nrow * _ncol );
}

template <typename T> inline void MatrixData<T>::resize( int nrow, int ncol )
{
  ASSERT(nrow>0 && ncol>0);
  _nrow = nrow;
  _ncol = ncol;
  delete [] _values;
  _values = new T[ _nrow * _ncol ];
}

template <typename T> inline bool MatrixData<T>::operator==( const MatrixData<T>& m ) const
{
  if( _nrow!=m._nrow || _ncol!=m._ncol ) return false;
  for (int i=_nrow * _ncol-1; i>=0;i--) {
    if (_values[i]!=m._values[i]) return false;
  }
  return true;
}

template <typename T> inline void MatrixData<T>::zero()
{
  memset( _values, 0, sizeof(T)*_nrow * _ncol );
}

template <typename T> inline void MatrixData<T>::identity()
{
  zero();
  int offset = _ncol + 1;
  int n=_nrow * _ncol;
  for (int i=0; i<n;i+=offset) _values[i]=1.0;
}

template <typename T> inline T& MatrixData<T>::at( int row, int col )
{
  ASSERT(row>=0 && row <_nrow);
  ASSERT(col>=0 && col <_ncol);
  return _values[ col * _nrow + row ];
}

template <typename T> inline const T& MatrixData<T>::at( int row, int col ) const
{
  ASSERT(row>=0 && row <_nrow);
  ASSERT(col>=0 && col <_ncol);
  return _values[ col * _nrow + row ];
}

template <typename T> inline void MatrixData<T>::print() const
{
  printf("   ");
  for (int j=0; j<_ncol; j++) {
    printf(" %6i", j);
  }
  printf("\n");
  for (int i=0; i<_nrow; i++) {
    printf("%3i", i);
    for (int j=0; j<_ncol; j++) {
      printf(" %6lf", Number::toDouble(at(i, j)));
    }
    printf("\n");
  }
}

} // namespace QGpCoreTools

#endif // MATRIXDATA_H
