/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2008-03-28
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef NUMBER_H
#define NUMBER_H

#include "QGpCoreToolsDLLExport.h"
#include "Trace.h"
#include "DateTime.h"
#include "Complex.h"

namespace QGpCoreTools {

#ifdef BIGNUM
#include <mp/mpreal.h>
#endif

class QGPCORETOOLS_EXPORT Number
{
public:
  enum Type {Fixed = 0, Scientific, Time};

  static inline Type type(char t);
  static inline char type(Type t);
  static inline double toDouble( QString str, Type t, bool * ok = 0 );
  static inline QString toString( double val, Type t, int precision );
  static double toDouble(float f);
  static double toDouble(const Complex & c) {return c.abs();}
  static double toDouble(double a) {return a;}
#ifdef BIGNUM
  static double toDouble( mp_real a ) { return dble( a );}
#endif

  static QList<int> divisors( int n );
};

inline Number::Type Number::type(char t)
{
  switch(t) {
  case 'e':
    return Scientific;
  case 't':
    return Time;
  default:
    return Fixed;
  }
}

inline char Number::type(Type t)
{
  switch(t) {
  case Fixed:
    break;
  case Scientific:
    return 'e';
  case Time:
    return 't';
  }
  return 'f';
}

inline double Number::toDouble( QString str, Type t, bool * ok )
{
  TRACE;
  switch(t) {
  case Fixed:
  case Scientific:
    return str.toDouble( ok );
  case Time:
    return DateTime::timeToSeconds( str );
  }
  return 0.0;
}

inline QString Number::toString( double val, Type t, int precision )
{
  TRACE;
  switch(t) {
  case Fixed:
    return QString::number( val, 'f', precision );
  case Scientific:
    return QString::number( val, 'e', precision );
  case Time:
    return DateTime::secondsToTime( val, precision );
  }
  return QString::null;
}

} // namespace QGpCoreTools

#endif // NUMBER_H
