/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2007-06-20
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PARALLELLOOP_H
#define PARALLELLOOP_H

#include <time.h>

#include "QGpCoreToolsDLLExport.h"
#include "Thread.h"

namespace QGpCoreTools {

class LoopTask;

class QGPCORETOOLS_EXPORT ParallelLoop : public QObject
{
  Q_OBJECT
public:
  ParallelLoop( QObject * parent = 0 ) : QObject(parent) {}

  void start(int iStart, int iEnd, bool forceParallel = false);
  void terminate();
  void waitFinished();

protected:
  virtual LoopTask * newTask() = 0;
protected slots:
  virtual void taskFinished();
private slots:
  void statusChanged(QString msg);
  void progressChanged(int value);
  void progressInit(int maximumValue);
signals:
  void statusChanged( int processIndex, QString msg );
  void progressChanged( int processIndex, int progress );
  void progressInit( int processIndex, int maximumValue);
  void finished();
private:
  inline LoopTask * newTask( int iEnd );
  struct TaskInfo {
    int index;
    int maximumProgress;
  };
  QMap<LoopTask *, TaskInfo> _tasks;
  QMutex _indexMutex;
  int _index;
};

class QGPCORETOOLS_EXPORT LoopTask : public Thread
{
  Q_OBJECT
public:
  LoopTask() : Thread() { _terminated = false; }

  void terminate() { _terminated.fetchAndStoreOrdered(true); }
  bool terminated() { return _terminated.testAndSetOrdered(true,true); }

  void setIndex(int * index, QMutex * mutex) {_index = index; _indexMutex = mutex;}
  void setEndIndex(int index) {_endIndex = index;}
protected:
  virtual void run();
  virtual void run(int index)=0;
  void setProgressValue( int value );
  void setProgressMaximum( int value ) { emit progressInit( value ); }
  void setStatus( QString msg ) { emit statusChanged( msg ); }
  int endIndex() const {return _endIndex;}
signals:
  void statusChanged( QString msg );
  void progressInit( int maximumValue );
  void progressChanged( int value );
private:
  QAtomicInt _terminated;
  QMutex * _indexMutex;
  int * _index;
  int _endIndex;
  QAtomicInt _lastProgressTime;
};

inline void LoopTask::setProgressValue( int value )
{
  TRACE;
  int now = time(0);
  if ( _lastProgressTime.fetchAndStoreOrdered(now) != now ) {
    progressChanged( value );
  }
}

} // namespace QGpCoreTools

#endif // PARALLELLOOP_H
