/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2002-04-15
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef POINT2D_H
#define POINT2D_H

#include <QtCore>

#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

class StringSection;

class QGPCORETOOLS_EXPORT Point2D
{
public:
  inline Point2D();
  inline Point2D(double x, double y);
  inline Point2D(const QPoint& p);
  inline Point2D(const QPointF& p);
  inline Point2D(const Point2D& p);

  double x() const {return _x;}
  double y() const {return _y;}
  void setX(double v) {_x=v;}
  void setY(double v) {_y=v;}
  void setValid(bool) {}
  bool isValid() const {return true;}
  void average(const Point2D& p) {_y= 0.5*(_y+p._y);}

  void degreesToDMS();
  void DMSToDegrees();
  void geographicalToRectangular(const Point2D& ref, double rotation=0.0);
  void rectangularToGeographical(const Point2D& ref, double rotation=0.0);
  void geographicalToUTM(QString * zone=0);
  QPoint utmZoneIndex() const;
  static QString utmZone(QPoint z);
  static double utmReference(QPoint z);

  // Copy operators
  inline Point2D& operator=(const Point2D& p);
  inline void set(double xi, double yi);

  // Comparison operators
  inline bool operator< (const Point2D& p) const;
  inline bool operator== (const Point2D& p) const;
  bool operator<= (const Point2D& obj) const
    { return (*this == obj || *this < obj);}
  bool operator>  (const Point2D& obj) const
    {return (! (*this <= obj));}
  bool operator>= (const Point2D& obj) const
    {return (! (*this < obj));}
  bool operator!= (const Point2D& obj) const
    {return (! (*this == obj));}
  inline short compare (const Point2D& p) const;

  // I/O functions
  bool fromString(const StringSection& str);
  QString toString(int precision=6) const;

  // Arithmetic operation
  inline void operator+=(const Point2D& p);
  inline void operator-=(const Point2D& p);
  inline void operator*=(const Point2D& p);
  inline void operator*=(double mul);
  inline void operator/=(const Point2D& p);
  inline void operator/=(double div);
  inline double scalarProduct(const Point2D& p) const;
  inline void interpole(double valX, const Point2D& p1, const Point2D& p2);
  void translate(const Point2D& p) {operator+(p);}
  void rotate(double angle);

  inline Point2D operator+(const Point2D& p) const;
  inline Point2D operator-(const Point2D& p) const;
  inline Point2D operator*(double mul) const;
  inline Point2D operator*(const Point2D& p) const;
  inline Point2D operator/(double div) const;
  inline Point2D operator/(const Point2D& p) const;

  // Miscellaneous
  bool isHit(const Point2D &p,double tolX, double tolY) const;
  double distanceTo(const Point2D &p) const;
  double distanceToSegment(const Point2D &p1, const Point2D &p2) const;
  double azimuthTo(const Point2D &p) const;
  double geographicalDistanceTo(const Point2D &p) const;
  double geographicalAzimuthTo(const Point2D &p) const;
  void move(double azimuth,double distance);
private:
  // Coordinates of the point
  double _x, _y;
};

} // namespace QGpCoreTools

namespace QGpCoreTools {

inline Point2D::Point2D()
{
  _x=0.;
  _y=0.;
}

inline Point2D::Point2D(double xi, double yi)
{
  _x=xi;
  _y=yi;
}

inline Point2D::Point2D(const QPoint& p)
{
  _x=p.x();
  _y=p.y();
}

inline Point2D::Point2D(const QPointF& p)
{
  _x=p.x();
  _y=p.y();
}

inline Point2D::Point2D(const Point2D& p)
{
  _x=p.x();
  _y=p.y();
}

inline Point2D& Point2D::operator= (const Point2D& p)
{
  _x=p._x;
  _y=p._y;
  return *this;
}

inline void Point2D::set(double xi, double yi)
{
  _x=xi;
  _y=yi;
}

inline bool Point2D::operator<  (const Point2D& p) const
{
  if (_x < p._x) return true;
  else if (_x==p._x && _y<p._y) return true;
  return false;
}

inline bool Point2D::operator== (const Point2D& p) const
{
  if ((_x==p._x) &&
      (_y==p._y)) return true;
  return false;
}

inline short Point2D::compare (const Point2D& p) const
{
  if (_x<p._x) return(-1);
  else if (_x>p._x) return(1);
  else if (_y<p._y) return(-1);
  else if (_y>p._y) return(1);
  else return(0);
}

inline Point2D Point2D::operator+(const Point2D& p) const
{
  return Point2D(_x+p._x, _y+p._y);
}

inline Point2D Point2D::operator-(const Point2D& p) const
{
  return Point2D (_x-p._x, _y-p._y);
}

inline Point2D Point2D::operator*(double mul) const
{
  return Point2D (_x*mul, _y*mul);
}

inline Point2D Point2D::operator*(const Point2D& p) const
{
  return Point2D (_x*p._x, _y*p._y);
}

inline Point2D Point2D::operator/(double div) const
{
  return Point2D (_x/div, _y/div);
}

inline Point2D Point2D::operator/(const Point2D& p) const
{
  return Point2D (_x/p._x, _y/p._y);
}

inline void Point2D::operator+=(const Point2D& p)
{
  _x+=p._x;
  _y+=p._y;
}

inline void Point2D::operator-=(const Point2D& p)
{
  _x-=p._x;
  _y-=p._y;
}

inline void Point2D::operator*=(const Point2D& p)
{
  _x*=p._x;
  _y*=p._y;
}

inline void Point2D::operator*=(double mul)
{
  _x*=mul;
  _y*=mul;
}

inline void Point2D::operator/=(const Point2D& p)
{
  _x/=p._x;
  _y/=p._y;
}

inline void Point2D::operator/=(double div)
{
  _x/=div;
  _y/=div;
}

inline double Point2D::scalarProduct(const Point2D& p) const
{
  return _x*p._x+_y*p._y;
}

inline void Point2D::interpole(double valX, const Point2D& p1, const Point2D& p2)
{
  double factor = ( valX - p1.x() ) / ( p2.x()-p1.x() );
  setX( valX );
  setY( p1.y()+ ( p2.y()-p1.y() ) * factor );
}

QGPCORETOOLS_EXPORT QTextStream& operator<< (QTextStream& s, const QList<Point2D>& plist);
QGPCORETOOLS_EXPORT QTextStream& operator>> (QTextStream& s, QList<Point2D>& plist);
QGPCORETOOLS_EXPORT QTextStream& operator<< (QTextStream& s, const Point2D& p);
QGPCORETOOLS_EXPORT QTextStream& operator>> (QTextStream& s, Point2D& p);
QGPCORETOOLS_EXPORT QDataStream& operator<< (QDataStream& s, const Point2D& p);
QGPCORETOOLS_EXPORT QDataStream& operator>> (QDataStream& s, Point2D& p);

} // namespace QGpCoreTools

#endif // POINT2D.H
