/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2008-11-09
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef POINTND_H
#define POINTND_H

#include "Trace.h"
#include "StringSection.h"
#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

class QGPCORETOOLS_EXPORT PointND
{
public:
  PointND() { _x=0.0; }
  PointND( double x, const QVector<double>& y ) { _x = x; _y = y; }
  PointND( const PointND& o ) { _x = o._x; _y = o._y; }

  int count() const { return _y.count(); }

  bool operator<(const PointND& p) const { return _x<p._x; }
  inline bool operator==(const PointND& p) const;
  inline void operator=(const PointND& p);

  double x() const { return _x; }
  void setX(double x) { _x = x; }
  double y() const { return y(0); }
  void setY( double y ) { setY(0, y); }
  inline double y( int index ) const;
  inline void setY( int index, double y );

  void setValid(bool) {}
  bool isValid() const { return true; }
  inline void interpole( double valX, const PointND& p1, const PointND& p2 );
  inline void average(const PointND&);

  // I/O functions
  bool fromString(const StringSection& str);
  QString toString(int precision=6) const;
private:
  double _x;
  QVector<double> _y;
};

inline bool PointND::operator==(const PointND& p) const
{
  if(_x!=p._x) return false;
  int n=_y.count();
  if (n!=p._y.count()) return false;
  for(int i = 0; i< n; i++) {
    if(_y.at(i)!=p._y.at(i)) return false;
  }
  return true;
}

inline void PointND::operator=(const PointND& p)
{
  _x=p._x;
  _y=p._y;
}

inline double PointND::y(int index) const
{
  TRACE;
  if ( index >= _y.count() ) {
    if ( _y.isEmpty() ) return 0.0; else return _y.last();
  } else {
    return _y[index];
  }
}

inline void PointND::setY(int index, double y)
{
  TRACE;
  if ( index >= _y.count() ) {
    _y.resize( index+1 );
  }
  _y[index] = y;
}

inline void PointND::interpole( double valX, const PointND& p1, const PointND& p2 )
{
  TRACE;
  double factor = ( valX - p1.x() ) / ( p2.x()-p1.x() );
  setX( valX );
  int n = p1.count();
  if (n<p2.count()) n = p2.count();
  for (int i = 0; i< n; i++) {
    setY( i, p1.y(i)+ ( p2.y(i)-p1.y(i) ) * factor );
  }
}

inline void PointND::average(const PointND& p)
{
  TRACE;
  int n = p.count();
  if (n<count()) n = count();
  for (int i = 0; i< n; i++) {
    setY( i, 0.5*( y(i) + p.y(i) ) );
  }
}

} // namespace QGpCoreTools

#endif // POINTND_H
