/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2006-09-25
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef STATPOINT_H
#define STATPOINT_H

#include <math.h>

#include "StatValue.h"
#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

class QGPCORETOOLS_EXPORT StatPoint : public StatValue
{
public:
  StatPoint() {}
  StatPoint(double x, double y)
    : StatValue(y), _x(x) {}

  virtual const QString& xml_tagName() const { return xmlStatPointTag;}
  static const QString xmlStatPointTag;

  inline void operator=( const StatPoint& o );
  inline void operator=( const StatValue& o );

  bool operator<(const StatPoint& p) const {return _x < p._x || ( _x == p._x && mean()<p.mean()); }
  bool operator==(const StatPoint& p) const {return _x == p._x && StatValue::operator==(p); }

  void setX( double v ) { _x = v;}
  double x() const {return _x;}

  void setY( double v ) { setMean(v);}
  double y() const {return mean();}

  inline void interpole( double valX, const StatPoint& p1, const StatPoint& p2 );

  // I/O functions
  bool fromString(StringSection str);
  QString toString(int precision=6) const;
protected:
  void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  XMLMember xml_member(XML_MEMBER_ARGS);
  bool xml_setProperty(XML_SETPROPERTY_ARGS);
private:
  double _x;
};

inline void StatPoint::operator=( const StatPoint& o )
{
  TRACE;
  StatValue::operator=( o );
  _x = o._x;
}

inline void StatPoint::operator=( const StatValue& o )
{
  TRACE;
  StatValue::operator=( o );
}

inline void StatPoint::interpole( double valX, const StatPoint& p1, const StatPoint& p2 )
{
  TRACE;
  double factor = ( valX - p1.x() ) / ( p2.x()-p1.x() );
  setX( valX );
  setMean( p1.mean()+ ( p2.mean()-p1.mean() ) * factor );
  setStddev( p1.stddev()+ ( p2.stddev()-p1.stddev() ) * factor );
  setWeight( p1.weight()+ ( p2.weight()-p1.weight() ) * factor );
}

} // namespace QGpCoreTools

#endif // STATPOINT_H
