/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-21
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef STATVALUE_H
#define STATVALUE_H

#include <math.h>

#include "XMLClass.h"
#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

class Value;

class QGPCORETOOLS_EXPORT StatValue : public XMLClass
{
public:
  inline StatValue();
  inline StatValue ( double mean, double stddev = 0.0, double weight = 1.0, bool valid = true );

  virtual const QString& xml_tagName() const { return xmlStatValueTag;}
  static const QString xmlStatValueTag;

  inline void operator=( const StatValue& o );
  inline bool operator==( const StatValue& o ) const;

  void setMean(double v) { _mean = v;}
  void setStddev(double v) { _stddev = v;}
  void setWeight(double v) {_weight=v; if(_weight==0.0) _valid=false;}

  double mean() const { return _mean;}
  double stddev() const { return _stddev;}
  double weight() const { return _weight;}

  void setValid( bool f ) {_valid = f; if(_valid && _weight==0.0) _weight=1.0;}
  bool isValid() const { return _valid; }

  enum MisfitType { L1, L1_Normalized, L1_LogNormalized, L1_NormalizedBySigmaOnly,
                    L2, L2_Normalized, L2_LogNormalized, L2_NormalizedBySigmaOnly,
                    Akaike, AkaikeFewSamples };
  static QString misfitTypeString( MisfitType type );
  static MisfitType misfitType( QString type );

  double misfit( int& nValues, int& nData, const Value& value, MisfitType type, double min ) const;

  inline void average( const StatValue& p );
protected:
  double _mean;
  double _stddev;
  double _weight;
  bool _valid;
protected:
  virtual void xml_writeProperties( XML_WRITEPROPERTIES_ARGS ) const;
  virtual XMLMember xml_member( XML_MEMBER_ARGS );
  virtual bool xml_setProperty( XML_SETPROPERTY_ARGS );
};

inline StatValue::StatValue()
{
  TRACE;
  _mean = 0.0;
  _stddev = 0.0;
  _weight = 1.0;
  _valid = true;
}

inline StatValue::StatValue( double mean, double stddev, double weight, bool valid )
{
  TRACE;
  _mean = mean;
  _stddev = stddev;
  _weight = weight;
  _valid = valid;
}

inline void StatValue::operator=( const StatValue& o )
{
  TRACE;
  _mean = o._mean;
  _stddev = o._stddev;
  _weight = o._weight;
  _valid = o._valid;
}

inline bool StatValue::operator==( const StatValue& o ) const
{
  return _mean==o._mean && _stddev == o._stddev && _weight == o._weight && _valid == o._valid;
}

inline void StatValue::average(const StatValue& o )
{
  TRACE;
  if(isValid()) {
    if (o.isValid()) {
      double mu2 = _mean*_mean;
      double wt = _weight + o._weight;
      _mean = ( _weight*_mean + o._weight*o._mean ) / wt;
      double sigma2 = _weight * mu2;
      if (_weight>1.0)
        sigma2 += ( _weight -1.0 )* _stddev*_stddev;
      else
        sigma2 += _weight* _stddev*_stddev;
      sigma2 += o._weight * o._mean * o._mean;
      if (o._weight>1.0)
        sigma2 += (o._weight-1.0) * o._stddev*o._stddev;
      else
        sigma2 += o._weight * o._stddev*o._stddev;
      sigma2 -= _mean*_mean * wt;
      sigma2 /= wt-1;
      _stddev = ::sqrt(sigma2);
      _weight = wt;
    }
  } else if(o.isValid()) {
    *this = o;
  }
}

QGPCORETOOLS_EXPORT QDataStream& operator<< ( QDataStream& s, const StatValue& p );
QGPCORETOOLS_EXPORT QDataStream& operator>> ( QDataStream& s, StatValue& p );

class QGPCORETOOLS_EXPORT StatValueVector : public QVector<StatValue>,
      public XMLClass
{
public:
  QVector<double> * meanVector() const;
  QVector<double> * stddevVector() const;
  QVector<double> * weightVector() const;
protected:
  virtual const QString& xml_tagName() const { return xmlStatValueVectorTag;}
  static const QString xmlStatValueVectorTag;
  virtual void xml_writeChildren( XML_WRITECHILDREN_ARGS ) const;
  virtual XMLMember xml_member( XML_MEMBER_ARGS );
};

} // namespace QGpCoreTools

#endif // STATVALUE_H
