/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-10-21
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef StringSection_H
#define StringSection_H


#include "Trace.h"
#include "QGpCoreToolsDLLExport.h"

namespace QGpCoreTools {

class QGPCORETOOLS_EXPORT StringSection
{
public:
  inline StringSection();
  inline StringSection(const QString& s);
  inline StringSection(const QChar * s);
  inline StringSection(const QChar * s, int len);
  inline StringSection(const QChar * s1, const QChar * s2);
  inline StringSection(const StringSection& o);
  ~StringSection() {}
  inline StringSection& operator=(const StringSection& o);

  inline bool operator==(const QChar * o) const;
  inline bool operator!=(const QChar * o) const;
  bool operator<(const QChar * o) const { return compare(o) < 0;}
  bool operator<=(const QChar * o) const { return compare(o) <= 0;}
  bool operator>(const QChar * o) const { return compare(o) > 0;}
  bool operator>=(const QChar * o) const { return compare(o) >= 0;}
  inline int compare(const QChar * o) const;

  inline bool operator==(const StringSection& o) const;
  inline bool operator!=(const StringSection& o) const;
  bool operator<(const StringSection& o) const { return compare(o) < 0;}
  bool operator<=(const StringSection& o) const { return compare(o) <= 0;}
  bool operator>(const StringSection& o) const { return compare(o) > 0;}
  bool operator>=(const StringSection& o) const { return compare(o) >= 0;}
  int compare(const StringSection &o) const;

  inline bool operator==(const QString& o) const;
  inline bool operator!=(const QString& o) const;
  bool operator<(const QString& o) const { return compare(o) < 0;}
  bool operator<=(const QString& o) const { return compare(o) <= 0;}
  bool operator>(const QString& o) const { return compare(o) > 0;}
  bool operator>=(const QString& o) const { return compare(o) >= 0;}
  inline int compare(const QString& o) const;

  QChar operator[] (int i) const {return _str[ i ];}
  void operator++() {_str++;if (_size > 0) _size--;}
  void operator+=(int l) {_str += l;if ((int) _size >= l) _size -= l; else _size = 0;}
  void operator--() {_str--;_size++;}

  bool isValid() const { return _str;}
  bool isEmpty() const { return _size == 0;}
  const QChar * data() const { return _str;}
  int size() const { return _size;}

  void shrink(int nChar) {if (nChar >= _size) _size = 0; else _size -= nChar;}
  StringSection& trimmed();
  void setInvalid() {_str = 0; _size = 0;}
  void set(const QString& s) { _str = s.data(); _size = s.size(); }
  void set(const QChar * s, int size) { _str = s; _size = size; }
  void setEnd(const QChar * e) { _size = e - _str; if (_size < 0) _size = 0; }
  void setSize(int size) {_size = size;}

  bool toBool() const {return *this=="true";}
  QString toString() const { return isValid() ? QString(_str, _size) : QString::null; }
  int toInt(bool * ok = 0, int base = 10) const { return toString().toInt(ok, base);}
  uint toUInt(bool * ok = 0, int base = 10) const { return toString().toUInt(ok, base);}
  qlonglong toLongLong(bool * ok = 0, int base = 10) const { return toString().toLongLong(ok, base);}
  double toDouble(bool * ok = 0) const { return toString().toDouble(ok);}
  void appendTo(QString& str) const { str.insert(str.size(), _str, _size); }

  // Force conversion to a type given by a fake argument (used by XMLMap)
  int to(int) const { return toInt(); }
  bool to(bool) const { return toBool(); }
  double to(double) const { return toDouble(); }
  QString to(QString) const { return toString(); }

  inline StringSection left(int n) const;
  inline StringSection right(int n) const;
  inline StringSection mid(int position, int n) const;

  const QChar * find(QChar c) const;
  int find(const StringSection target, bool reverse) const;
  bool contains(const StringSection target) const {return find(target, false)>-1;}
  bool contains(QChar target) const {return find(StringSection(&target, 1), false)>-1;}
  inline bool beginWith(const QString& o) const;
  inline bool endWith(const QString& o) const;
  bool beginWith(const StringSection& o) const;
  bool endWith(const StringSection& o) const;

  StringSection readLine(const QChar *& ptr) const;
  bool nextLine(const QChar *& ptr, const QString& sep=defaultSeparators) const;
  StringSection nextField(const QChar *& ptr, const QString& sep=defaultSeparators, bool skipEmpty=true) const;

  static int size(const QChar * str);
  static int compare(const QChar * str1, const QChar * str2, int n);
  static const QString defaultSeparators;
  static const StringSection null;
private:
  const QChar * _str;
  int _size;
};

QGPCORETOOLS_EXPORT uint qHash(StringSection key);

inline StringSection::StringSection()
{
  _str = 0;
  _size = 0;
}

inline StringSection::StringSection(const QString& s)
{
  _str = s.data();
  _size = s.size();
}

inline StringSection::StringSection(const QChar * s)
{
  _str = s;
  _size = size(s);
}

inline StringSection::StringSection(const QChar * s1, const QChar * s2)
{
  _str = s1;
  setEnd(s2);
}

inline StringSection::StringSection(const QChar * s, int len)
{
  _str = s;
  _size = len;
}

inline StringSection::StringSection(const StringSection& o)
{
  *this = o;
}

inline StringSection& StringSection::operator=(const StringSection& o)
{
  _str = o._str;
  _size = o._size;
  return *this;
}

inline int StringSection::compare(const QChar * o) const
{
  StringSection os(o);
  return compare(os);
}

inline int StringSection::compare(const QString& o) const
{
  StringSection os(o);
  return compare(os);
}

inline bool StringSection::operator==(const QChar * o) const
{
  if (_size != size(o)) return false;
  return compare(o, _str, _size) == 0;
}

inline bool StringSection::operator==(const StringSection& o) const
{
  if (_size != o._size) return false;
  return compare(o._str, _str, _size) == 0;
}

inline bool StringSection::operator==(const QString& o) const
{
  if (_size != o.size()) return false;
  return compare(o.data(), _str, _size) == 0;
}

inline bool StringSection::operator!=(const QChar * o) const
{
  if (_size != size(o)) return true;
  return compare(o, _str, _size) != 0;
}

inline bool StringSection::operator!=(const StringSection& o) const
{
  if (_size != o._size) return true;
  return compare(o._str, _str, _size) != 0;
}

inline bool StringSection::operator!=(const QString& o) const
{
  if (_size != o.size()) return true;
  return compare(o.data(), _str, _size) != 0;
}

inline bool StringSection::beginWith(const StringSection& o) const
{
  if (_size < o._size) return false;
  return compare(o._str, _str, o.size()) == 0;
}

inline bool StringSection::endWith(const StringSection& o) const
{
  if (_size < o._size) return false;
  return compare(o._str, _str+_size-o._size, o._size) == 0;
}

inline bool StringSection::beginWith(const QString& o) const
{
  if (_size < o.size()) return false;
  return compare(o.data(), _str, o.size()) == 0;
}

inline bool StringSection::endWith(const QString& o) const
{
  if (_size < o.size()) return false;
  return compare(o.data(), _str+_size-o.size(), o.size()) == 0;
}

inline StringSection StringSection::left(int n) const
{
  if(n>_size) n=_size;
  return StringSection(_str, n);
}

inline StringSection StringSection::right(int n) const
{
  if(n>_size) n=_size;
  return StringSection(_str+_size-n, n);
}

inline StringSection StringSection::mid(int position, int n) const
{
  if(position>=_size) return StringSection::null;
  int ns=_size-position;
  if(n<ns) ns=n;
  return StringSection(_str+position, ns);
}

} // namespace QGpCoreTools

#endif // STRINGSECTION_H
