/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-04-28
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef LAYEREDMODEL_H
#define LAYEREDMODEL_H

#include <QGpCoreTools.h>

#include "QGpCoreWaveDLLExport.h"
#include "Profile.h"

namespace QGpCoreWave {

class QGPCOREWAVE_EXPORT LayeredModel
{
  TRANSLATIONS("LayeredModel")
public:
  LayeredModel();
  LayeredModel( const LayeredModel& o );
  LayeredModel( int layersNumber );
  ~LayeredModel();

  enum SlownessType { P, S };

  void operator=( const LayeredModel& o );

  // Function used for computing dispersion curves
  double maxSlowR() const { return _maxRayleighSlowness; }
  double minSlowS() const { return _slowSMin; }
  double maxSlowS() const { return _slowSMax; }
  bool initCalculation();
  int checkVelocityInversion() const;

  void setH( int iLayer, double v ) { _h[iLayer] = v; }
  void setSlowP( int iLayer, double v ) { _slowP[iLayer] = v; }
  void setSlowS( int iLayer, double v ) { _slowS[iLayer] = v; }
  void setRho( int iLayer, double v ) { _rho[iLayer] = v; }
  void setQp( int iLayer, double v ) { _qp[iLayer] = v; }
  void setQs( int iLayer, double v ) { _qs[iLayer] = v; }

  // Layer informations
  int layerCount() const { return _layerCount; }
  double h( int iLayer ) const { return _h[iLayer]; }
  double slowP( int iLayer ) const { return _slowP[iLayer]; }
  double slowS( int iLayer ) const { return _slowS[iLayer]; }
  double rho( int iLayer ) const { return _rho[iLayer]; }
  double mu( int iLayer ) const { return _mu[iLayer]; }
  double qp( int iLayer ) const { return _qp[iLayer]; }
  double qs( int iLayer ) const { return _qs[iLayer]; }
  double poisson( int iLayer ) const;
  double depth( int iLayer ) const;
  double expGrad( int iLayer ) const;
  double halfSpaceRayleighSlowness( int iLayer ) const;
  bool elastic() const;

  double vpAt( double depth ) const;
  double vsAt( double depth ) const;
  double rhoAt( double depth ) const;
  double poissonAt( double depth ) const;
  double impedanceAt( double depth ) const;
  double slowPAt( double depth ) const;
  double slowSAt( double depth ) const;
  double roughFundamentalFrequency() const;

  Profile slowpProfile() const;
  inline Profile vpProfile() const;
  Profile slowsProfile() const;
  inline Profile vsProfile() const;
  Profile rhoProfile() const;
  Profile poissonProfile() const;
  Profile impedanceProfile() const;

  // Format model in a multiline string
  static QString formatHelp();
  QString toString() const;
  bool fromStream( QTextStream& s, QString * comments = 0 );
  // Prints the model parameters to the stream s
  void toStream( QTextStream& s ) const;
  // Prints the model parameters to the stream s
  void exportHerrmann( QTextStream& s ) const;
  // Prints the model parameters to the stream s as binary
  void toStream( QDataStream& s ) const;
  // Load the model parameters from the stream s as binary
  void fromStream(QDataStream& s);

  /* Modify one thickness keeping all other depth constant (if possible)
    if not, modify only the thickness of this layer and other depth will also change*/
  void changeHKeepOtherDepth(int iLayer,double newH);
  // Change Vp without changing the Poisson ratio of this layer
  void changeVpKeepPoisson(int iLayer,double newV);
  // Change Vs without changing the Poisson ratio of this layer
  void changeVsKeepPoisson(int iLayer,double newV);
  // Copy H values from another model (only if same number of layers)
  void setHFrom( const LayeredModel& o );
protected:
  void allocateData();
  void deleteData();
private:
  // number of layers including the bottom half space
  int _layerCount;
  // pointer to thickness vector
  double *_h;
  // pointer to compressional slownesses vector
  double *_slowP;
  // pointer to shear slownesses vector
  double *_slowS;
  // pointer to densities vector
  double *_rho;
  // pointer to mu vector (calculated)
  double *_mu;
  // Quality factors
  double *_qp;
  double *_qs;
  /* Calculated Wave Number of an homogeneous half space with the maximum shear slowness */
  double _maxRayleighSlowness;
  double _slowSMin;
  double _slowSMax;

  bool setValue( double& var, const StringSection& field, int iLayer, const QString& varName, bool optional );
  void initMembers();
  int checkPoissonRatio();
  void slowMinMax( int& iSlowSMax);
};

inline Profile LayeredModel::vpProfile() const
{
  Profile p = slowpProfile();
  p.inverse();
  return p;
}

inline Profile LayeredModel::vsProfile() const
{
  Profile p = slowsProfile();
  p.inverse();
  return p;
}

} // namespace QGpCoreWave

#endif // LAYEREDMODEL_H
