/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2006-09-25
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MODALSTORAGE_H
#define MODALSTORAGE_H

#include <QGpCoreTools.h>
#include "ModalRefine.h"
#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class QGPCOREWAVE_EXPORT ModalStorage
{
public:
  ModalStorage();
  ModalStorage( int nModes, const QVector<double> * x ) { set( nModes, x ); }
  ~ModalStorage() { delete [] _values; }

  void set( int nModes, const QVector<double> * x );
  ModalStorage& operator=( const ModalStorage& o );

  void log10();
  void exp10();
  void abs();

  int xCount() const { return _nX; }
  int modeCount() const { return _nModes; }
  double x( int ix ) const { return ( *_x ) [ ix ]; }
  Value * mode( int index ) { return _values + index * _nX; }
  const Value * mode( int index ) const { return _values + index * _nX; }

  int refineCount() const { return _refines.count();}
  ModalRefine& refineAdd( double x ) {
    _refines.append( ModalRefine( _nModes, x ) );
    _refines.last().setValid(false);
    return _refines.last();
  }
  void refineClear() {_refines.clear();}
  void refineSort() {qSort( _refines );}
  ModalRefine& lastRefine() { return _refines.last();}
  RefineIterator refineBegin() const { return _refines.begin(); }
  RefineIterator refineEnd() const { return _refines.end(); }

  void toStream(QTextStream& s, int modeIndex =-1) const;
  Curve<Point2D> curve( int iMode ) const;

  void writeReport( QDataStream& s ) const;

  class Iterator {
  public:
    inline Iterator( const ModalStorage * s, int iMode );
    inline bool atEnd() const;
    inline Iterator &operator++();
    inline double x() const;
    inline const Value& value();
  private:
    const ModalStorage * _storage;
    double _nextFixed, _nextRefine;
    int _fixedIterator;
    RefineIterator _refineIterator;
    int _iMode;
    const Value * _values;
  };

private:
  int _nX, _nModes;
  const QVector<double> * _x;
  Value * _values;
  RefineList _refines;
};

inline ModalStorage::Iterator::Iterator( const ModalStorage * s, int iMode )
{
  TRACE;
  _storage = s;
  _nextFixed = _storage->x( 0 );
  _fixedIterator = 0;
  _refineIterator = _storage->refineBegin();
  if ( _refineIterator != _storage->refineEnd() ) {
    _nextRefine = _refineIterator->x();
  } else {
    _nextRefine = 1e99;
  }
  _iMode = iMode;
  _values = _storage->mode( _iMode );
}

inline ModalStorage::Iterator& ModalStorage::Iterator::operator++()
{
  if ( _nextFixed < _nextRefine ) {
    _fixedIterator++;
    if ( _fixedIterator < _storage->xCount() )
      _nextFixed = _storage->x( _fixedIterator );
    else {
      _nextFixed = 1e99;
    }
  } else {
    ++_refineIterator;
    if ( _refineIterator != _storage->refineEnd() )
      _nextRefine = _refineIterator->x();
    else {
      _nextRefine = 1e99;
    }
  }
  return *this;
}

inline bool ModalStorage::Iterator::atEnd() const
{
  TRACE;
  return _fixedIterator == _storage->xCount() && _refineIterator == _storage->refineEnd();
}

inline double ModalStorage::Iterator::x() const
{
  TRACE;
  if ( _nextFixed < _nextRefine ) {
    return _nextFixed;
  } else {
    return _nextRefine;
  }
}

inline const Value& ModalStorage::Iterator::value()
{
  if ( _nextFixed < _nextRefine ) {
    return _values[_fixedIterator] ;
  } else {
    return _refineIterator->value( _iMode ) ;
  }
}

} // namespace QGpCoreWave

#endif // MODALSTORAGE_H
