/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2003-02-13
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef GRAPHICSHEET_H
#define GRAPHICSHEET_H

#include <QtGui>

#include "SelectionWidget.h"

namespace SciFigs {

class GraphicObject;
class SelectionWidget;
class AxisWindow;
class GraphContent;
class TextEdit;
class OrderTip;
class GraphicSheetProperties;

class SCIFIGS_EXPORT GraphicSheet : public QScrollArea, public PropertyContext, public PropertyItem, public XMLClass
{
  Q_OBJECT
  // Overload to mark the beginning of properties to save to XML
  Q_PROPERTY( QString objectName READ objectName WRITE setObjectName STORED true)
  // Kept only to avoid warning on loading old files
  Q_PROPERTY( QString name READ dummyPropertyString WRITE setObjectName STORED false )
  Q_PROPERTY( QString objectList READ dummyPropertyString WRITE setDummyProperty STORED false )
  Q_PROPERTY( QString objectOrder READ dummyPropertyString WRITE setDummyProperty STORED false )
  // Use only in scripts
  Q_PROPERTY( bool orderIndex READ isOrderIndex WRITE showOrderIndex STORED false SCRIPTABLE true)
  Q_PROPERTY( int objectCount READ objectCount STORED false SCRIPTABLE true )
  Q_PROPERTY( int transparency READ transparency WRITE setTransparency SCRIPTABLE true )
  Q_PROPERTY( bool transparentMask READ transparentMask WRITE setTransparentMask SCRIPTABLE true )
  DUMMY_PROPERTIES;
public:
  GraphicSheet( QWidget *parent = 0 );
  ~GraphicSheet();

  virtual const QString& xml_tagName() const {return xmlGraphicSheetTag;}
  static const QString xmlGraphicSheetTag;

  QString uniqueName( GraphicObject * obj, QString newName );
  int objectCount();
  void clear( bool force );
  QString objectList() const;
  void setObjectList( QString list );
  void setObjectOrder( QString list );
  // Count the palette currently in the sheet
  int paletteCount();
  bool removeObject( GraphicObject * obj, bool force );
  void addObject( GraphicObject * obj );
  GraphicObject * findObject( QString tag, QString name );
  void showObject( GraphicObject * obj );
  void showAllObjects();
  // Set status bar where to show messages
  void setStatusBar( QStatusBar * s ) {_statusBar = s;}
  QStatusBar * statusBar() { return _statusBar;}
  bool isOrderIndex() const { return _orderIndex >= 0;}
  void setOrderIndex( GraphicObject * obj );
  // Caluclate max right and bottom coordinate of child graphs and resize content
  void autoResizeContent();
  double printBottom();
  double printRight();
  // Find maximum printer resolution in all graphs (!! in dpi !!)
  int maximumResolution();
  template <typename T> QList<T> selectedObjects() const;
  GraphicObject * activeObject( const char * inherits = 0 );
  void print( QPrinter& printer );
  Point2D currentOrigin();

  void addFileActions( QMenu * m, QToolBar * tb );
  void addEditActions( QMenu * m, QToolBar * tb );
  void addFormatActions( QMenu * m );
  void addInsertActions( QMenu * m, QToolBar * tb );
  void addMenu( QMenu * m );

  SelectionWidget * addSelection( GraphicObject * obj );

  void addProperties( PropertyProxy * pp );
  void removeProperties( PropertyProxy * pp );
  virtual void properties( PropertyWidget * w ) const;
  virtual void setProperty( uint wid, int pid, QVariant val );

  void setTransparency( int val ) { _transparency = val; }
  void setTransparentMask( bool val );

  int transparency() const { return _transparency;}
  bool transparentMask() const { return _transparentMask;}

  virtual QSize minimumSizeHind() const { return QSize(100,100); }
protected:
  virtual void mousePressEvent ( QMouseEvent * e );
  virtual void mouseReleaseEvent ( QMouseEvent * e );
  void addActions();
private:
  SelectionWidget * _currentActiveSelection;
  void printSvg( const QString& fileName, int dpi );
  void printSize( double& x0, double& y0, double& x1, double& y1,
                  double& mx0, double& my0, double& mx1, double& my1 );

  bool _transparentMask;
  int _transparency;

  int _orderIndex;
  // Smartly remember the last loaded file
  QString _currentFile;
  // Smoother drag engine
  Point2D _dragDelta;
  QTimer _dragDelay;
  /* Pointer to a status bar where to show messages
   (allways check if non null before using it!!) */
  QStatusBar * _statusBar;
  // List of objects added during xml restore
  QList<GraphicObject *> _loadedObjects;

  static uint _category, _tabPrint;
signals:
  void currentFileChanged( const QString& fileName );
  void activeSelectionChanged( GraphicObject * obj );
private slots:
  void dragObjects( Point2D delta );
  void dragObjectsDelayed();
  QList<GraphicObject *> graphicObjects() const;
public slots:
  Q_SCRIPTABLE void clear() { clear(false); }
  Q_SCRIPTABLE void fileNew();
  Q_SCRIPTABLE void fileOpen();
  Q_SCRIPTABLE void fileOpen( QString fileName, double dx = 0, double dy = 0 );
  Q_SCRIPTABLE void fileSave();
  Q_SCRIPTABLE void fileSaveAs();
  Q_SCRIPTABLE void fileSave( QString fileName );
  Q_SCRIPTABLE void setPreferences ();

  Q_SCRIPTABLE void undo();
  Q_SCRIPTABLE void redo();
  Q_SCRIPTABLE void cut();
  Q_SCRIPTABLE void copy();
  Q_SCRIPTABLE void copyImage();
  Q_SCRIPTABLE void paste();

  Q_SCRIPTABLE void selectAll();
  Q_SCRIPTABLE bool selectAll( QString actionTitle );
  Q_SCRIPTABLE void unSelectAll() {selectObjects(0, Qt::NoModifier);}
  Q_SCRIPTABLE void selectObjects( GraphicObject * obj, Qt::KeyboardModifiers m = Qt::NoModifier );
  Q_SCRIPTABLE void select(GraphicObject * obj) {selectObjects(obj, Qt::NoModifier);}
  Q_SCRIPTABLE void addSelect(GraphicObject * obj)  {selectObjects(obj, Qt::ShiftModifier);}
  Q_SCRIPTABLE void selectNextChild();
  Q_SCRIPTABLE void addSelectNextChild();

  GraphicObject * addObject();
  Q_SCRIPTABLE bool removeObject( GraphicObject * obj ) {return removeObject (obj, false); }
  Q_SCRIPTABLE GraphicObject * object(QString objectName);
  Q_SCRIPTABLE GraphicObject * object(int index);

  void moveObject( GraphicObject * obj = 0 );
  void resizeObject();
  void startDragging();
  Q_SCRIPTABLE void exportFile( QString fileName = QString::null, QString fileFormat = QString::null, int dpi = 0 );
  Q_SCRIPTABLE void exportImage( QString fileName = QString::null, QString imageFormat = QString::null, int dpi = 0 );
  Q_SCRIPTABLE QPixmap image( int dpi = 0 );
  Q_SCRIPTABLE void print( int dpi = 0 );
  Q_SCRIPTABLE void print(bool outputToFile, QString name, int dpi);
  void updatePageLimits( QPrinter& p );
  Q_SCRIPTABLE void copyMakeUp();
  Q_SCRIPTABLE void pasteMakeUp();
  Q_SCRIPTABLE void saveMakeUp( QString fileName = QString::null );
  Q_SCRIPTABLE void restoreMakeUp( QString fileName = QString::null );
  Q_SCRIPTABLE void saveLayers( QString fileName = QString::null );
  Q_SCRIPTABLE void addLayers();
  virtual Q_SCRIPTABLE void update();
  virtual Q_SCRIPTABLE void deepUpdate();
  Q_SCRIPTABLE void showOrderIndex( bool isOn );
  Q_SCRIPTABLE void alignLeft() {alignX( 0.0 );}
  Q_SCRIPTABLE void alignHCenter() {alignX( 0.5 );}
  Q_SCRIPTABLE void alignRight() {alignX( 1.0 );}
  Q_SCRIPTABLE void alignTop() {alignY( 0.0 );}
  Q_SCRIPTABLE void alignVCenter() {alignY( 0.5 );}
  Q_SCRIPTABLE void alignBottom() {alignY( 1.0 );}
  Q_SCRIPTABLE void alignX( double localPosition );
  Q_SCRIPTABLE void alignY( double localPosition );
  Q_SCRIPTABLE void raiseSelection();
  Q_SCRIPTABLE void lowerSelection();
  Q_SCRIPTABLE void showProperties();
protected:
  virtual void xml_polish(XML_POLISH_ARGS);
  virtual void xml_polishChild(XML_POLISHCHILD_ARGS);
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS) {return qobject_setProperty( this, memberID, attributes, content, context );}
};

template <typename T>
QList<T> GraphicSheet::selectedObjects() const
{
  QList<T> wList;
  QList<SelectionWidget *> wSelList = findChildren<SelectionWidget *>();
  SelectionWidget * wSel;
  foreach( wSel, wSelList ) {
    if(wSel->state()!=SelectionWidget::Hoover) {
      GraphicObject * w = wSel->widget();
      T castedW = qobject_cast<T>( w );
      if ( castedW )
        wList << castedW;
    }
  }
  return wList;
}

} // namespace SciFigs

#endif // GRAPHICSHEET_H
