/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2004-12-01
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PEN_H
#define PEN_H

#include <QtGui>
#include <QGpCoreTools.h>
#include "SciFigsDLLExport.h"

namespace SciFigs {

class SCIFIGS_EXPORT Pen : public XMLClass
{
public:
  inline Pen();
  inline Pen( Qt::PenStyle style );
  inline Pen( const QColor& col, double width = 0, Qt::PenStyle style = Qt::SolidLine );
  inline Pen( const QColor& col, double width, Qt::PenStyle s, Qt::PenCapStyle c,
                 Qt::PenJoinStyle j );
  inline Pen( const Pen& o );
  virtual ~Pen() {}

  virtual const QString& xml_tagName() const { return xmlPenTag;}
  static const QString xmlPenTag;

  inline QPen qpen( double dotpercm ) const;
  inline Pen& operator=( const Pen& o );
  inline bool operator==( const Pen& o );

  Qt::PenStyle style() const { return _style;}
  void setStyle( Qt::PenStyle s ) {_style = s;}
  double width() const { return _width;}
  int screenWidth( double dotpercm ) const
  {
    return ( int ) floor( 0.1 * _width * dotpercm + 0.5 );
  }
  void setWidth ( double w ) {_width = w;}
  const QColor& color() const { return _color;}
  void setColor( const QColor & c ) {_color = c;}
  Qt::PenCapStyle capStyle() const { return _capStyle;}
  void setCapStyle( Qt::PenCapStyle c ) {_capStyle = c;}
  Qt::PenJoinStyle joinStyle() const { return _joinStyle;}
  void setJoinStyle( Qt::PenJoinStyle j ) {_joinStyle = j;}

  static Qt::PenStyle styleValue( QString s );
  static QString styleName( Qt::PenStyle s );
  static Qt::PenCapStyle capStyleValue( QString s );
  static QString capStyleName( Qt::PenCapStyle s );
  static Qt::PenJoinStyle joinStyleValue( QString s );
  static QString joinStyleName( Qt::PenJoinStyle s );
  static const Pen null;
protected:
  virtual void xml_writeProperties( XML_WRITEPROPERTIES_ARGS ) const;
  virtual bool xml_setProperty( XML_SETPROPERTY_ARGS );
  virtual XMLMember xml_member( XML_MEMBER_ARGS );
private:
  QColor _color;
  Qt::PenStyle _style;
  Qt::PenCapStyle _capStyle;
  Qt::PenJoinStyle _joinStyle;
  // in mm
  double _width;
};

// SCIFIGS_EXPORT QTextStream& operator>>(QTextStream& s,Pen& p);

inline Pen::Pen()
{
  _color = Qt::black;
  _style = Qt::SolidLine;
  _capStyle = Qt::FlatCap;
  _joinStyle = Qt::MiterJoin;
  _width = 0.3;
}

inline Pen::Pen( Qt::PenStyle style )
{
  _color = Qt::black;
  _style = style;
  _capStyle = Qt::FlatCap;
  _joinStyle = Qt::MiterJoin;
  _width = 0.3;
}

inline Pen::Pen( const QColor& col, double width, Qt::PenStyle style )
{
  _color = col;
  _style = style;
  _capStyle = Qt::FlatCap;
  _joinStyle = Qt::MiterJoin;
  _width = width;
}

inline Pen::Pen( const QColor& col, double width, Qt::PenStyle s, Qt::PenCapStyle c,
                       Qt::PenJoinStyle j )
{
  _color = col;
  _style = s;
  _capStyle = c;
  _joinStyle = j;
  _width = width;
}

inline Pen::Pen( const Pen& o ) : XMLClass()

{
  _color = o._color;
  _style = o._style;
  _capStyle = o._capStyle;
  _joinStyle = o._joinStyle;
  _width = o._width;
}

inline Pen& Pen::operator=( const Pen& o )
{
  _color = o._color;
  _style = o._style;
  _capStyle = o._capStyle;
  _joinStyle = o._joinStyle;
  _width = o._width;
  return *this;
}

inline bool Pen::operator==( const Pen& o )
{
  TRACE;
  return _color == o._color
      && _style == o._style
      && _capStyle == o._capStyle
      && _joinStyle == o._joinStyle
      && _width == o._width;
}

inline QPen Pen::qpen( double dotpercm ) const
{
  TRACE;
  return QPen( QBrush( _color ), round( 0.1 * _width * dotpercm ), _style, _capStyle,
               _joinStyle );
}

} // namespace SciFigs

#endif // PEN_H
