/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2006-07-25
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SCALE_H
#define SCALE_H

#include <QGpCoreTools.h>
#include "SciFigsDLLExport.h"

namespace SciFigs {

class SCIFIGS_EXPORT Scale
{
public:
  Scale();

  enum Type {Linear, Inversed, Log};

  void operator=(const Scale& o);

  Type type() const {return _type;}
  void setType( Type t ) { _type = t;}

  SamplingOption sampling() const;

  void setGlobalMinimum( double m ) { _globalMinimum = m; }
  double globalMinimum() const { return _globalMinimum; }

  void setGlobalMaximum( double m ) { _globalMaximum = m; }
  double globalMaximum() const { return _globalMaximum; }

  void autoTicks( Number::Type numberType );

  double majorTicks() const { return _majorTicks; }
  void setMajorTicks( double m ) { _majorTicks = m; }

  double minorTicks() const { return _minorTicks; }
  void setMinorTicks( double m ) { _minorTicks = m; }

  double minimum() const {return _minimum;}
  void setMinimum( double min ) { _minimum = min; }

  double maximum() const {return _maximum;}
  void setMaximum( double max ) { _maximum = max;}

  double a() const {return _a;}
  double b() const {return _b;}

  bool isReversed() const {return _reversed;}
  void setReversed(bool r) {_reversed=r;}
  inline bool isEffectivelyReversed() const;

  inline double s2r( int val ) const;
  inline int r2s( double val ) const;

  void setHorizontalTransformation( int length );
  void setVerticalTransformation( int length );

  inline int lineCount() const;
  void setVerticalCurrentLine( int line);
  int verticalCurrentLine() const;
  void setHorizontalCurrentLine( int line);
  int horizontalCurrentLine() const;

  void checkLimits();
private:
  inline double a2r(double val) const;
  inline double r2a(double val) const;

  double _a, _b;  // Current parameters for tranformation between screen and real coordinates
  Type _type;
  double _globalMinimum, _globalMaximum;
  double _minimum, _maximum;  // Current visible minimum and maximum
  bool _reversed;
  // increment for labels and primary ticks
  double _majorTicks;
  // increment for secondary ticks
  double _minorTicks;
};

inline int Scale::r2s( double val ) const
{
  TRACE;
  switch ( _type ) {
  case Inversed:
    return ( int ) floor( _a / val + _b + 0.5 );
  case Log:
    return ( int ) floor( _a * log10( val ) + _b + 0.5 );
  default:
    return ( int ) floor( _a * val + _b + 0.5 );
  }
}

inline double Scale::s2r( int val ) const
{
  TRACE;
  switch ( _type ) {
  case Inversed:
    return _a / ( (double) val - _b );
  case Log:
    return pow( 10.0, ( (double) val - _b ) / _a );
  default:
    return ( (double) val -_b ) / _a;
  }
}

inline double Scale::r2a( double val ) const
{
  TRACE;
  switch ( _type ) {
  case Inversed:
    return 1.0 / val;
  case Log:
    return log10( val );
  default:
    return val;
  }
}

inline double Scale::a2r( double val ) const
{
  TRACE;
  switch ( _type ) {
  case Inversed:
    return 1.0 / val;
  case Log:
    return pow( 10.0, val );
  default:
    return val;
  }
}

inline bool Scale::isEffectivelyReversed() const
{
  TRACE;
  return _reversed ? _type != Inversed : _type == Inversed;
}

inline int Scale::lineCount() const
{
  TRACE;
  // Line is one tenth of physical size of axis in pixel (aSize()*0.1). Hence 9 steps are removed
  // because the screen is always full of content.
  return (int) ceil((r2a(_globalMaximum) - r2a(_globalMinimum))/((r2a(_maximum) - r2a(_minimum))*0.1))-9;
}

} // namespace SciFigs

#endif // SCALE_H
