/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created : 2002-05-07
**  Authors :
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef XYPLOT_H
#define XYPLOT_H

#include "GraphicObject.h"
#include "GraphContentLayer.h"

namespace SciFigs {

class XYPlotProperties;

class SCIFIGS_EXPORT XYPlot : public GraphContentLayer
{
  Q_OBJECT
  Q_PROPERTY( double dotDiameter READ dotDiameter WRITE setDotDiameter SCRIPTABLE true )
  Q_PROPERTY( int count READ count STORED false SCRIPTABLE true )
public:
  XYPlot( AxisWindow * parent = 0 );
  ~XYPlot();

  virtual const QString& xml_tagName() const { return xmlXYPlotTag;}
  static const QString xmlXYPlotTag;

  int count() const {if (_xData) return _xData->count(); else return 0;}

  void setXData( QVector<double> * x );
  void setYData( QVector<double> * y );

  // Diameter of point on plot (in cm)
  double dotDiameter() const { return _size;}
  // Change diameter of point on plot (in cm)
  void setDotDiameter ( double d ) {_size = d;}

  virtual bool trackRectangle( int, double rx1, double ry1, double rx2, double ry2,
                               Qt::KeyboardModifiers eventState );
  virtual void select( QVector<int> * indexList );
  // Calculate the rectangle that includes all the point
  virtual Rect boundingRect() const;

  virtual bool hasProperties() { return true; }
  virtual void addProperties( PropertyProxy * pp );
  virtual void removeProperties( PropertyProxy * pp );
  virtual void properties( PropertyWidget * w ) const;
  virtual void setProperty( uint wid, int pid, QVariant val );
public slots:
  void swapXY();
protected:
  virtual void paintData( const LayerPainterRequest& lp, QPainter& p, double dotpercm ) const;
  inline void getPaintParam(int& minSize, int& pointSize, int& halfSize, double dotpercm) const;

  QVector<double> * _xData;
  QVector<double> * _yData;

  // Diameter of point on plot (in cm)
  double _size;

  static uint _tab;

  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual void xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const;
  virtual bool xml_setBinaryData(XML_SETBINARYDATA_ARGS);
};

inline void XYPlot::getPaintParam(int& minSize, int& pointSize, int& halfSize, double dotpercm) const
{
  TRACE;
  minSize = _xData->size();
  if ( minSize > _yData->size() ) minSize = _yData->size();
  double pointSizeF=_size * dotpercm;
  if (pointSizeF<=2.0) {
    pointSize = 2;
    halfSize = 1;
  } else {
    pointSize = ( int ) floor( pointSizeF + 0.5 );
    halfSize = ( int ) floor( 0.5 * pointSizeF+0.5 );
  }
}

} // namespace SciFigs

#endif // XYPLOT_H
